"""
API Processing Lambda - Handles IoT Wireless device operations
"""
import boto3
import os
import json
import logging
import time
from typing import Dict, Any, List
from datetime import datetime
from botocore.exceptions import ClientError

logger = logging.getLogger()
logger.setLevel(logging.INFO)

# Initialize AWS clients with retry configuration
from botocore.config import Config

# Configure retry behavior for IoT Wireless client
retry_config = Config(
    retries={
        'max_attempts': 5,
        'mode': 'adaptive'  # Uses adaptive retry mode with exponential backoff
    }
)

iot_wireless_client = boto3.client('iotwireless', config=retry_config)
s3_client = boto3.client('s3')

cloudwatch = boto3.client('cloudwatch')
rds_client = boto3.client('rds-data')

def execute_sql(sql, parameters=None):
    """Execute SQL using RDS Data API"""
    try:
        request = {
            'resourceArn': os.environ['DB_CLUSTER_ARN'],
            'secretArn': os.environ['DB_SECRET_ARN'],
            'database': os.environ['DB_NAME'],
            'sql': sql
        }
        
        if parameters:
            request['parameters'] = parameters
            
        response = rds_client.execute_statement(**request)
        logger.info(f"SQL executed successfully. Rows affected: {response.get('numberOfRecordsUpdated', 'unknown')}")
        return response
        
    except Exception as e:
        logger.error(f"Failed to execute SQL: {str(e)}")
        logger.error(f"SQL: {sql}")
        if parameters:
            logger.error(f"Parameters: {parameters}")
        # Don't raise the exception to avoid breaking the main workflow
        return None

def emit_metric(metric_name: str, value: float, unit: str = 'Count', dimensions: Dict[str, str] = None):
    """Emit CloudWatch metric"""
    try:
        metric_data = {
            'MetricName': metric_name,
            'Value': value,
            'Unit': unit,
            'Timestamp': datetime.utcnow()
        }
        
        if dimensions:
            metric_data['Dimensions'] = [
                {'Name': k, 'Value': v} for k, v in dimensions.items()
            ]
        
        cloudwatch.put_metric_data(
            Namespace='IoTWireless/BulkManagement',
            MetricData=[metric_data]
        )
        
    except Exception as e:
        logger.error(f"Failed to emit metric {metric_name}: {str(e)}")
        # Don't raise the exception to avoid breaking the main workflow

def extract_main_batch_name(batch_name: str) -> str:
    """
    Extract main batch name from individual batch name
    Example: 'gamma-loadtest-iad-create-devices-all-7' -> 'gamma-loadtest-iad-create-devices-all'
    """
    if batch_name and '-' in batch_name:
        parts = batch_name.rsplit('-', 1)
        if len(parts) == 2 and parts[1].isdigit():
            return parts[0]
    return batch_name

def store_device_in_database(device: Dict[str, Any], batch_name: str, aws_device_id: str = None, status: str = 'PROCESSING'):
    """Store device information in database using RDS Data API"""
    try:
        # Validate device parameter type
        if not isinstance(device, dict):
            logger.error(f"Device parameter must be a dictionary, got {type(device)}: {device}")
            return
        
        # Extract main batch name (remove individual batch number suffix)
        main_batch_name = extract_main_batch_name(batch_name)
        if main_batch_name != batch_name:
            logger.info(f"Extracted main batch name: {main_batch_name} from individual batch: {batch_name}")
        
        sql = """
            INSERT INTO devices (
                smsn, aws_wireless_device_id, device_name, device_profile_id,
                uplink_destination_name, batch_id, status, status_details,
                positioning_enabled, positioning_destination_name, created_at, updated_at
            ) VALUES (
                :smsn, :aws_wireless_device_id, :device_name, :device_profile_id,
                :uplink_destination_name, :batch_id, :status, :status_details,
                :positioning_enabled, :positioning_destination_name, CURRENT_TIMESTAMP, CURRENT_TIMESTAMP
            )
            ON CONFLICT (smsn) DO UPDATE SET
                aws_wireless_device_id = COALESCE(EXCLUDED.aws_wireless_device_id, devices.aws_wireless_device_id),
                device_name = CASE WHEN EXCLUDED.device_name != '' THEN EXCLUDED.device_name ELSE devices.device_name END,
                device_profile_id = CASE WHEN EXCLUDED.device_profile_id != '' THEN EXCLUDED.device_profile_id ELSE devices.device_profile_id END,
                uplink_destination_name = CASE WHEN EXCLUDED.uplink_destination_name != '' THEN EXCLUDED.uplink_destination_name ELSE devices.uplink_destination_name END,
                batch_id = EXCLUDED.batch_id,
                status = EXCLUDED.status,
                status_details = EXCLUDED.status_details,
                positioning_enabled = EXCLUDED.positioning_enabled,
                positioning_destination_name = CASE WHEN EXCLUDED.positioning_destination_name != '' THEN EXCLUDED.positioning_destination_name ELSE devices.positioning_destination_name END,
                updated_at = CURRENT_TIMESTAMP
        """
        
        # Handle both camelCase and snake_case field names
        device_name = device.get('deviceName') or device.get('device_name', '')
        device_profile_id = device.get('deviceProfileId') or device.get('device_profile_id', '')
        uplink_destination_name = device.get('uplinkDestinationName') or device.get('uplink_destination_name', '')
        
        # Handle null values properly for RDS Data API
        aws_device_param = {'name': 'aws_wireless_device_id'}
        if aws_device_id:
            aws_device_param['value'] = {'stringValue': aws_device_id}
        else:
            aws_device_param['value'] = {'isNull': True}
        
        parameters = [
            {'name': 'smsn', 'value': {'stringValue': device.get('smsn', '')}},
            aws_device_param,
            {'name': 'device_name', 'value': {'stringValue': device_name}},
            {'name': 'device_profile_id', 'value': {'stringValue': device_profile_id}},
            {'name': 'uplink_destination_name', 'value': {'stringValue': uplink_destination_name}},
            {'name': 'batch_id', 'value': {'stringValue': main_batch_name}},
            {'name': 'status', 'value': {'stringValue': status}},
            {'name': 'status_details', 'value': {'stringValue': ''}},
            {'name': 'positioning_enabled', 'value': {'booleanValue': device.get('positioning', {}).get('enabled', False)}},
            {'name': 'positioning_destination_name', 'value': {'stringValue': device.get('positioning', {}).get('positioningDestinationName', '')}}
        ]
        
        response = execute_sql(sql, parameters)
        
        if response:
            rows_affected = response.get('numberOfRecordsUpdated', 0)
            logger.info(f"Stored device {device.get('smsn')} in database with status {status} ({rows_affected} rows affected)")
        else:
            logger.error(f"Failed to get response from database insert for device {device.get('smsn')}")
        
    except Exception as e:
        logger.error(f"Failed to store device in database: {str(e)}")

def update_device_status(smsn: str, status: str, aws_device_id: str = None, error_message: str = None):
    """Update device status in database using RDS Data API"""
    try:
        logger.info(f"Updating device {smsn} status to {status} with aws_device_id: {aws_device_id}")
        
        sql = """
            UPDATE devices 
            SET status = :status,
                aws_wireless_device_id = :aws_wireless_device_id,
                status_details = :status_details,
                updated_at = CURRENT_TIMESTAMP
            WHERE smsn = :smsn
        """
        
        # Handle null values properly for RDS Data API
        aws_device_param = {'name': 'aws_wireless_device_id'}
        if aws_device_id:
            aws_device_param['value'] = {'stringValue': aws_device_id}
            logger.info(f"Setting aws_device_id to: {aws_device_id}")
        else:
            aws_device_param['value'] = {'isNull': True}
            logger.info(f"Setting aws_device_id to NULL (device creation failed or not available)")
        
        parameters = [
            {'name': 'status', 'value': {'stringValue': status}},
            aws_device_param,
            {'name': 'status_details', 'value': {'stringValue': error_message or ''}},
            {'name': 'smsn', 'value': {'stringValue': smsn}}
        ]
        
        response = execute_sql(sql, parameters)
        
        if response:
            rows_updated = response.get('numberOfRecordsUpdated', 0)
            if rows_updated > 0:
                logger.info(f"Successfully updated device {smsn} status to {status} ({rows_updated} rows affected)")
            else:
                logger.warning(f"No rows updated for device {smsn} - device may not exist in database")
                # Try to check if device exists
                check_sql = "SELECT smsn, status FROM devices WHERE smsn = :smsn"
                check_params = [{'name': 'smsn', 'value': {'stringValue': smsn}}]
                check_response = execute_sql(check_sql, check_params)
                if check_response and check_response.get('records'):
                    logger.info(f"Device {smsn} exists with current status: {check_response['records'][0]}")
                else:
                    logger.error(f"Device {smsn} not found in database")
        else:
            logger.error(f"Failed to get response from database update for device {smsn}")
            
    except Exception as e:
        logger.error(f"Failed to update device status: {str(e)}")

def get_device_smsn(aws_device_id: str) -> tuple:
    """Get SMSN and device name from AWS IoT Wireless device using GetWirelessDevice API"""
    try:
        logger.info(f"Getting SMSN for device ID: {aws_device_id}")
        
        response = retry_with_backoff(
            lambda: iot_wireless_client.get_wireless_device(
                Identifier=aws_device_id,
                IdentifierType='WirelessDeviceId'
            ),
            max_attempts=3,
            base_delay=0.5,
            max_delay=15.0
        )
        
        # Extract device name from response
        device_name = response.get('Name', '')
        
        # Extract SMSN from the response
        # For Sidewalk devices, SMSN is in the SidewalkManufacturingSn field
        if 'Sidewalk' in response and 'SidewalkManufacturingSn' in response['Sidewalk']:
            smsn = response['Sidewalk']['SidewalkManufacturingSn']
            logger.info(f"Found SMSN {smsn} and device name {device_name} for device ID {aws_device_id}")
            return smsn, device_name
        elif 'Name' in response:
            # Fallback to device name if SMSN not available
            logger.warning(f"SMSN not found, using device name as identifier: {response['Name']}")
            return response['Name'], device_name
        else:
            logger.warning(f"Could not extract SMSN from device {aws_device_id}")
            return aws_device_id, device_name  # Use device ID as fallback
            
    except Exception as e:
        logger.error(f"Failed to get SMSN for device {aws_device_id}: {str(e)}")
        return aws_device_id, ''  # Use device ID as fallback

def get_device_aws_id(smsn: str) -> tuple:
    """Get AWS wireless device ID and device name from SMSN using GetWirelessDevice API"""
    try:
        logger.info(f"Getting AWS device ID for SMSN: {smsn}")
        
        response = retry_with_backoff(
            lambda: iot_wireless_client.get_wireless_device(
                Identifier=smsn,
                IdentifierType='SidewalkManufacturingSn'
            ),
            max_attempts=3,
            base_delay=0.5,
            max_delay=15.0
        )
        
        # Extract device name from response
        device_name = response.get('Name', '')
        
        # Extract AWS device ID from the response
        # The device ID is in the 'Id' field of the response
        if 'Id' in response:
            aws_device_id = response['Id']
            logger.info(f"Found AWS device ID {aws_device_id} and device name {device_name} for SMSN {smsn}")
            return aws_device_id, device_name
        elif 'Sidewalk' in response and 'AmazonId' in response['Sidewalk']:
            # Alternative: use AmazonId if available
            aws_device_id = response['Sidewalk']['AmazonId']
            logger.info(f"Found AWS device ID (AmazonId) {aws_device_id} and device name {device_name} for SMSN {smsn}")
            return aws_device_id, device_name
        else:
            logger.warning(f"Could not extract AWS device ID from response for SMSN {smsn}")
            return None, device_name
            
    except Exception as e:
        logger.error(f"Failed to get AWS device ID for SMSN {smsn}: {str(e)}")
        return None, ''

def is_retryable_error(error: Exception) -> bool:
    """Determine if an error is retryable"""
    if isinstance(error, ClientError):
        error_code = error.response['Error']['Code']
        # Throttling and temporary errors are retryable
        if error_code in ['ThrottlingException', 'ServiceUnavailableException', 'InternalServerException']:
            return True
        # Validation and permanent errors are not retryable
        if error_code in ['ValidationException', 'ConflictException', 'ResourceNotFoundException']:
            return False
    
    # Network/connection errors are generally retryable
    if 'ConnectionError' in str(type(error)) or 'TimeoutError' in str(type(error)):
        return True
    
    return False

def retry_with_backoff(func, max_attempts=3, base_delay=1.0, max_delay=60.0, backoff_multiplier=2.0):
    """
    Retry a function with exponential backoff
    
    Args:
        func: Function to retry (should be a lambda or callable)
        max_attempts: Maximum number of retry attempts
        base_delay: Initial delay in seconds
        max_delay: Maximum delay in seconds
        backoff_multiplier: Multiplier for exponential backoff
    
    Returns:
        Result of the function call or raises the last exception
    """
    last_exception = None
    
    for attempt in range(max_attempts):
        try:
            return func()
        except Exception as e:
            last_exception = e
            
            # Don't retry if it's not a retryable error
            if not is_retryable_error(e):
                logger.warning(f"Non-retryable error on attempt {attempt + 1}: {str(e)}")
                raise e
            
            # Don't sleep on the last attempt
            if attempt < max_attempts - 1:
                # Calculate delay with exponential backoff
                delay = min(base_delay * (backoff_multiplier ** attempt), max_delay)
                logger.warning(f"Retryable error on attempt {attempt + 1}/{max_attempts}: {str(e)}. Retrying in {delay:.2f}s")
                time.sleep(delay)
            else:
                logger.error(f"Max attempts ({max_attempts}) reached. Last error: {str(e)}")
    
    # If we get here, all attempts failed
    raise last_exception

def validate_tps_configuration(operation: str, actual_tps: int):
    """
    Validate that the Lambda is receiving the correct TPS for the operation type
    """
    try:
        if operation == 'create':
            expected_tps = int(os.environ.get('CREATE_DEVICE_TPS', '10'))
            # For create operations, we expect the full TPS divided by concurrency
            create_max_concurrency = int(os.environ.get('CREATE_MAX_CONCURRENCY', '5'))
            expected_tps_per_lambda = max(1, expected_tps // create_max_concurrency)
            
            logger.info(f"Create operation - Expected TPS per lambda: {expected_tps_per_lambda}, Actual: {actual_tps}")
            
            if abs(actual_tps - expected_tps_per_lambda) > 1:  # Allow small rounding differences
                logger.warning(f"TPS mismatch for create operation. Expected: ~{expected_tps_per_lambda}, Got: {actual_tps}")
                
        else:  # update
            update_tps = int(os.environ.get('UPDATE_DEVICE_TPS', '10'))
            get_tps = int(os.environ.get('GET_DEVICE_TPS', '10'))
            expected_effective_tps = min(update_tps, get_tps)
            update_max_concurrency = int(os.environ.get('UPDATE_MAX_CONCURRENCY', '5'))
            expected_tps_per_lambda = max(1, expected_effective_tps // update_max_concurrency)
            
            logger.info(f"Update operation - Update TPS: {update_tps}, Get TPS: {get_tps}")
            logger.info(f"Update operation - Expected TPS per lambda: {expected_tps_per_lambda}, Actual: {actual_tps}")
            
            if abs(actual_tps - expected_tps_per_lambda) > 1:  # Allow small rounding differences
                logger.warning(f"TPS mismatch for update operation. Expected: ~{expected_tps_per_lambda}, Got: {actual_tps}")
                
        # Emit metric for TPS validation
        emit_metric('TPSValidation', actual_tps, 'Count', {
            'Operation': operation,
            'ExpectedTPS': str(expected_tps_per_lambda if 'expected_tps_per_lambda' in locals() else 'unknown')
        })
        
    except Exception as e:
        logger.error(f"Failed to validate TPS configuration: {str(e)}")
        # Don't raise exception - this is just validation

def create_wireless_device(device: Dict[str, Any], batch_name: str, api_delay: float) -> Dict[str, Any]:
    """Create a wireless device"""
    # Validate device parameter type
    if not isinstance(device, dict):
        logger.error(f"Device parameter must be a dictionary, got {type(device)}: {device}")
        return {
            'smsn': str(device) if device else 'unknown',
            'status': 'FAILED',
            'error_code': 'InvalidDeviceData',
            'error_message': f'Device data must be a dictionary, got {type(device)}',
            'retryable': False,
            'timestamp': datetime.utcnow().isoformat()
        }
    
    smsn = device.get('smsn')
    
    try:
        # Store device in database with PROCESSING status
        store_device_in_database(device, batch_name, status='PROCESSING')
        
        # Handle both camelCase and snake_case field names
        device_name = device.get('deviceName') or device.get('device_name')
        device_profile_id = device.get('deviceProfileId') or device.get('device_profile_id')
        uplink_destination_name = device.get('uplinkDestinationName') or device.get('uplink_destination_name', '')
        
        # Prepare create request
        create_request = {
            'Type': 'Sidewalk',
            'Name': device_name,
            'DestinationName': uplink_destination_name,
            'Sidewalk': {
                'SidewalkManufacturingSn': smsn,
                'DeviceProfileId': device_profile_id
            }
        }
        
        # Add positioning if specified
        positioning_config = device.get('positioning', {})
        if positioning_config.get('enabled'):
            create_request['Positioning'] = 'Enabled'
            
            # Add positioning destination within Sidewalk configuration
            if positioning_config.get('positioningDestinationName'):
                create_request['Sidewalk']['Positioning'] = {
                    'DestinationName': positioning_config['positioningDestinationName']
                }
        
        logger.info(f"Creating device {smsn} with name {device_name}")
        
        # Call IoT Wireless API with retry logic
        response = retry_with_backoff(
            lambda: iot_wireless_client.create_wireless_device(**create_request),
            max_attempts=3,
            base_delay=1.0,
            max_delay=30.0
        )
        aws_device_id = response['Id']
        
        logger.info(f"Successfully created device {smsn} with ID {aws_device_id}")
        
        # Emit success metrics
        emit_metric('DeviceCreated', 1, dimensions={'Operation': 'create'})
        emit_metric('APICallSuccess', 1, dimensions={'Operation': 'create', 'APICall': 'CreateWirelessDevice'})
        
        # Rate limiting
        if api_delay > 0:
            time.sleep(api_delay)
        
        # Update database with success
        update_device_status(smsn, 'PROVISIONED', aws_device_id)
        
        return {
            'smsn': smsn,
            'aws_wireless_device_id': aws_device_id,
            'status': 'SUCCESS',
            'timestamp': datetime.utcnow().isoformat()
        }
        
    except Exception as e:
        logger.error(f"Failed to create device {smsn}: {str(e)}")
        
        # Emit failure metrics
        emit_metric('DeviceProcessingFailure', 1, dimensions={'Operation': 'create'})
        emit_metric('APICallFailure', 1, dimensions={'Operation': 'create'})
        
        # Determine error code for metrics
        error_code = 'Unknown'
        if isinstance(e, ClientError):
            error_code = e.response['Error']['Code']
        
        emit_metric('DeviceProcessingFailure', 1, dimensions={
            'Operation': 'create',
            'ErrorCode': error_code
        })
        
        error_message = str(e)
        
        # Update database with failure (no aws_device_id since creation failed)
        update_device_status(smsn, 'FAILED', aws_device_id=None, error_message=error_message)
        
        return {
            'smsn': smsn,
            'status': 'FAILED',
            'error_code': error_code,
            'error_message': error_message,
            'retryable': is_retryable_error(e),
            'timestamp': datetime.utcnow().isoformat()
        }

def update_wireless_device(device: Dict[str, Any], batch_name: str, api_delay: float) -> Dict[str, Any]:
    """Update a wireless device"""
    # Validate device parameter type
    if not isinstance(device, dict):
        logger.error(f"Device parameter must be a dictionary, got {type(device)}: {device}")
        return {
            'smsn': str(device) if device else 'unknown',
            'status': 'FAILED',
            'error_code': 'InvalidDeviceData',
            'error_message': f'Device data must be a dictionary, got {type(device)}',
            'retryable': False,
            'timestamp': datetime.utcnow().isoformat()
        }
    
    smsn = device.get('smsn')
    aws_device_id = device.get('awsWirelessDeviceId') or device.get('aws_wireless_device_id')
    
    # Handle different scenarios for getting both SMSN and AWS device ID
    if not smsn and aws_device_id:
        # Case 1: AWS device ID provided, need to get SMSN and device name
        smsn, retrieved_device_name = get_device_smsn(aws_device_id)
        logger.info(f"Retrieved SMSN {smsn} for device ID {aws_device_id}")
        # Update the device dict with the retrieved SMSN for database storage
        device['smsn'] = smsn
        # If device name is not provided in the update request, use the retrieved one
        if not device.get('deviceName') and not device.get('device_name') and retrieved_device_name:
            device['deviceName'] = retrieved_device_name
            logger.info(f"Using retrieved device name: {retrieved_device_name}")
    elif smsn and not aws_device_id:
        # Case 2: SMSN provided, need to get AWS device ID and device name
        aws_device_id, retrieved_device_name = get_device_aws_id(smsn)
        if aws_device_id:
            logger.info(f"Retrieved AWS device ID {aws_device_id} for SMSN {smsn}")
            # Update the device dict with the retrieved AWS device ID for database storage
            device['awsWirelessDeviceId'] = aws_device_id
            # If device name is not provided in the update request, use the retrieved one
            if not device.get('deviceName') and not device.get('device_name') and retrieved_device_name:
                device['deviceName'] = retrieved_device_name
                logger.info(f"Using retrieved device name: {retrieved_device_name}")
        else:
            logger.error(f"Could not retrieve AWS device ID for SMSN {smsn}")
            return {
                'smsn': smsn,
                'aws_wireless_device_id': '',
                'status': 'FAILED',
                'error_code': 'MissingDeviceId',
                'error_message': f'Could not retrieve AWS wireless device ID for SMSN {smsn}',
                'retryable': False,
                'timestamp': datetime.utcnow().isoformat()
            }
    
    # Validate we have required identifiers
    if not smsn:
        logger.error(f"No SMSN available for device update (aws_device_id: {aws_device_id})")
        return {
            'smsn': aws_device_id or 'unknown',
            'aws_wireless_device_id': aws_device_id,
            'status': 'FAILED',
            'error_code': 'MissingIdentifier',
            'error_message': 'No SMSN provided and could not retrieve from AWS IoT Wireless',
            'retryable': False,
            'timestamp': datetime.utcnow().isoformat()
        }
    
    if not aws_device_id:
        logger.error(f"No AWS device ID available for device update (smsn: {smsn})")
        return {
            'smsn': smsn,
            'aws_wireless_device_id': '',
            'status': 'FAILED',
            'error_code': 'MissingDeviceId',
            'error_message': 'No AWS wireless device ID available for update operation',
            'retryable': False,
            'timestamp': datetime.utcnow().isoformat()
        }
    
    try:
        # Store device in database with PROCESSING status (now we have both SMSN and AWS device ID)
        store_device_in_database(device, batch_name, aws_device_id, status='PROCESSING')
        
        # Prepare update request
        update_request = {
            'Id': aws_device_id
        }
        
        # Handle both camelCase and snake_case field names
        device_name = device.get('deviceName') or device.get('device_name')
        uplink_destination_name = device.get('uplinkDestinationName') or device.get('uplink_destination_name')
        
        # Add fields to update
        if device_name:
            update_request['Name'] = device_name
        
        if uplink_destination_name:
            update_request['DestinationName'] = uplink_destination_name
        
        # Handle positioning updates
        positioning_config = device.get('positioning', {})
        if positioning_config.get('enabled') is not None:
            # Update positioning status
            update_request['Positioning'] = 'Enabled' if positioning_config.get('enabled') else 'Disabled'
            
            # If positioning is enabled and destination is specified, add within Sidewalk configuration
            if positioning_config.get('enabled') and positioning_config.get('positioningDestinationName'):
                update_request['Sidewalk'] = {
                    'Positioning': {
                        'DestinationName': positioning_config['positioningDestinationName']
                    }
                }
        
        logger.info(f"Updating device {smsn} with ID {aws_device_id}")
        
        # Call IoT Wireless API with retry logic
        retry_with_backoff(
            lambda: iot_wireless_client.update_wireless_device(**update_request),
            max_attempts=3,
            base_delay=1.0,
            max_delay=30.0
        )
        
        logger.info(f"Successfully updated device {smsn}")
        
        # Emit success metrics
        emit_metric('DeviceUpdated', 1, dimensions={'Operation': 'update'})
        emit_metric('APICallSuccess', 1, dimensions={'Operation': 'update', 'APICall': 'UpdateWirelessDevice'})
        
        # Rate limiting
        if api_delay > 0:
            time.sleep(api_delay)
        
        # Update database with success
        update_device_status(smsn, 'UPDATED', aws_device_id)
        
        return {
            'smsn': smsn,
            'aws_wireless_device_id': aws_device_id,
            'status': 'SUCCESS',
            'timestamp': datetime.utcnow().isoformat()
        }
        
    except Exception as e:
        logger.error(f"Failed to update device {smsn}: {str(e)}")
        
        # Emit failure metrics
        emit_metric('DeviceProcessingFailure', 1, dimensions={'Operation': 'update'})
        emit_metric('APICallFailure', 1, dimensions={'Operation': 'update'})
        
        # Determine error code for metrics
        error_code = 'Unknown'
        if isinstance(e, ClientError):
            error_code = e.response['Error']['Code']
        
        emit_metric('DeviceProcessingFailure', 1, dimensions={
            'Operation': 'update',
            'ErrorCode': error_code
        })
        
        error_message = str(e)
        
        # Update database with failure (keep existing aws_device_id for updates)
        update_device_status(smsn, 'FAILED', aws_device_id=aws_device_id, error_message=error_message)
        
        return {
            'smsn': smsn,
            'aws_wireless_device_id': aws_device_id,
            'status': 'FAILED',
            'error_code': error_code,
            'error_message': error_message,
            'retryable': is_retryable_error(e),
            'timestamp': datetime.utcnow().isoformat()
        }

def load_batch_from_s3(bucket: str, key: str) -> List[Dict]:
    """Load batch data from S3"""
    try:
        response = s3_client.get_object(Bucket=bucket, Key=key)
        batch_data = json.loads(response['Body'].read().decode('utf-8'))
        
        # Extract devices from the batch data
        devices = batch_data.get('devices', [])
        logger.info(f"Loaded {len(devices)} devices from S3 batch")
        return devices
        
    except Exception as e:
        logger.error(f"Failed to load batch from S3 s3://{bucket}/{key}: {str(e)}")
        raise

def log_tps_environment():
    """Log all TPS-related environment variables for debugging"""
    tps_vars = {
        'CREATE_DEVICE_TPS': os.environ.get('CREATE_DEVICE_TPS', 'NOT_SET'),
        'UPDATE_DEVICE_TPS': os.environ.get('UPDATE_DEVICE_TPS', 'NOT_SET'),
        'GET_DEVICE_TPS': os.environ.get('GET_DEVICE_TPS', 'NOT_SET'),
        'CREATE_MAX_CONCURRENCY': os.environ.get('CREATE_MAX_CONCURRENCY', 'NOT_SET'),
        'UPDATE_MAX_CONCURRENCY': os.environ.get('UPDATE_MAX_CONCURRENCY', 'NOT_SET'),
        'CREATE_TPS_PER_LAMBDA': os.environ.get('CREATE_TPS_PER_LAMBDA', 'NOT_SET'),
        'UPDATE_TPS_PER_LAMBDA': os.environ.get('UPDATE_TPS_PER_LAMBDA', 'NOT_SET')
    }
    
    logger.info("TPS Environment Variables:")
    for var, value in tps_vars.items():
        logger.info(f"  {var}: {value}")

def lambda_handler(event, context):
    """
    Main handler for API processing
    """
    logger.info("API processing started")
    logger.info(f"Event: {json.dumps(event)}")
    
    # Log TPS environment for debugging
    log_tps_environment()
    
    try:
        # Extract parameters from event
        operation = event.get('operation', 'create')
        batch_name = event.get('batchName', 'unknown')
        
        # Load devices from S3 if S3 reference is provided, otherwise use direct devices
        s3_bucket = event.get('s3Bucket')
        s3_key = event.get('s3Key')
        
        if s3_bucket and s3_key:
            # Load batch data from S3
            devices = load_batch_from_s3(s3_bucket, s3_key)
            logger.info(f"Loaded batch from S3: s3://{s3_bucket}/{s3_key}")
        else:
            # Fallback to direct devices parameter (for backward compatibility)
            devices_param = event.get('devices', [])
            devices = devices_param if isinstance(devices_param, list) else []
            logger.info(f"Using direct devices parameter: {len(devices)} devices")
        
        # Get TPS per lambda from event or calculate based on operation type
        tps_per_lambda = event.get('tpsPerLambda')
        if tps_per_lambda is None:
            # Fallback: calculate TPS based on operation type and environment variables
            if operation == 'create':
                create_tps = int(os.environ.get('CREATE_DEVICE_TPS', '10'))
                tps_per_lambda = create_tps  # Use full TPS for single lambda fallback
            else:  # update
                update_tps = int(os.environ.get('UPDATE_DEVICE_TPS', '10'))
                get_tps = int(os.environ.get('GET_DEVICE_TPS', '10'))
                # For updates, we need both get and update calls, so use the minimum
                effective_tps = min(update_tps, get_tps)
                tps_per_lambda = effective_tps
            
            logger.info(f"TPS per lambda not provided in event, calculated: {tps_per_lambda} for operation: {operation}")
        else:
            logger.info(f"Using TPS per lambda from event: {tps_per_lambda} for operation: {operation}")
        
        # Validate TPS value
        if tps_per_lambda <= 0:
            logger.warning(f"Invalid TPS value: {tps_per_lambda}, defaulting to 1")
            tps_per_lambda = 1
        
        # Validate that TPS matches expected values for operation type
        validate_tps_configuration(operation, tps_per_lambda)
        
        # Calculate API delay for rate limiting
        api_delay = 1.0 / tps_per_lambda if tps_per_lambda > 0 else 0
        
        logger.info(f"Processing {len(devices)} devices for operation: {operation}")
        logger.info(f"Batch: {batch_name}, TPS per Lambda: {tps_per_lambda}")
        
        # Emit batch processing metrics
        emit_metric('BatchProcessingStarted', 1, dimensions={'Operation': operation})
        emit_metric('DevicesInBatch', len(devices), dimensions={'Operation': operation})
        
        # Process devices
        successful_devices = []
        failed_devices = []
        
        for i, device in enumerate(devices):
            logger.info(f"Processing device {i}: type={type(device)}, value={device}")
            
            # Skip non-dictionary devices
            if not isinstance(device, dict):
                logger.error(f"Skipping invalid device {i}: expected dict, got {type(device)}")
                failed_devices.append({
                    'smsn': f'invalid-device-{i}',
                    'status': 'FAILED',
                    'error_code': 'InvalidDeviceFormat',
                    'error_message': f'Device must be a dictionary, got {type(device)}: {device}',
                    'retryable': False,
                    'timestamp': datetime.utcnow().isoformat()
                })
                continue
            
            try:
                if operation == 'create':
                    result = create_wireless_device(device, batch_name, api_delay)
                elif operation == 'update':
                    result = update_wireless_device(device, batch_name, api_delay)
                else:
                    raise ValueError(f"Unsupported operation: {operation}")
                
                if result['status'] == 'SUCCESS':
                    successful_devices.append(result)
                else:
                    failed_devices.append(result)
                    
            except Exception as e:
                logger.error(f"Unexpected error processing device {device.get('smsn')}: {str(e)}")
                failed_devices.append({
                    'smsn': device.get('smsn'),
                    'status': 'FAILED',
                    'error_code': 'UnexpectedError',
                    'error_message': str(e),
                    'retryable': False,
                    'timestamp': datetime.utcnow().isoformat()
                })
        
        # Calculate success rate
        total_devices = len(devices)
        success_count = len(successful_devices)
        failure_count = len(failed_devices)
        success_rate = (success_count / total_devices * 100) if total_devices > 0 else 0
        
        # Emit completion metrics
        emit_metric('BatchProcessingCompleted', 1, dimensions={'Operation': operation})
        emit_metric('BatchSuccessRate', success_rate, unit='Percent', dimensions={'Operation': operation})
        
        logger.info(f"API processing completed: {success_count} successful, {failure_count} failed")
        
        # Store detailed results in S3 to avoid Step Functions payload size limits
        detailed_results = {
            'successful_devices': successful_devices,
            'failed_devices': failed_devices,
            'batch_summary': {
                'operation': operation,
                'batch_name': batch_name,
                'total_devices': total_devices,
                'success_count': success_count,
                'failure_count': failure_count,
                'success_rate': success_rate,
                'timestamp': datetime.utcnow().isoformat()
            }
        }
        
        # Store detailed results in S3
        report_bucket = os.environ.get('REPORT_BUCKET')
        if report_bucket:
            try:
                results_key = f"batch-results/{batch_name}-results.json"
                s3_client.put_object(
                    Bucket=report_bucket,
                    Key=results_key,
                    Body=json.dumps(detailed_results),
                    ContentType='application/json'
                )
                logger.info(f"Stored detailed results in S3: s3://{report_bucket}/{results_key}")
                
                # Return lightweight summary for Step Functions
                return {
                    'statusCode': 200,
                    'operation': operation,
                    'batch_name': batch_name,
                    'total_devices': total_devices,
                    'success_count': success_count,
                    'failure_count': failure_count,
                    'success_rate': success_rate,
                    'results_s3_key': results_key,
                    'results_s3_bucket': report_bucket,
                    'timestamp': datetime.utcnow().isoformat()
                }
                
            except Exception as s3_error:
                logger.error(f"Failed to store results in S3: {str(s3_error)}")
                # Fall back to returning summary data only (no detailed device lists)
                return {
                    'statusCode': 200,
                    'operation': operation,
                    'batch_name': batch_name,
                    'total_devices': total_devices,
                    'success_count': success_count,
                    'failure_count': failure_count,
                    'success_rate': success_rate,
                    'timestamp': datetime.utcnow().isoformat(),
                    'note': 'Detailed results not stored due to S3 error'
                }
        else:
            logger.warning("REPORT_BUCKET not configured - returning summary only")
            # Return summary data only (no detailed device lists)
            return {
                'statusCode': 200,
                'operation': operation,
                'batch_name': batch_name,
                'total_devices': total_devices,
                'success_count': success_count,
                'failure_count': failure_count,
                'success_rate': success_rate,
                'timestamp': datetime.utcnow().isoformat(),
                'note': 'Detailed results not stored - REPORT_BUCKET not configured'
            }
        
    except Exception as e:
        logger.error(f"API processing failed: {str(e)}")
        emit_metric('BatchProcessingFailure', 1)
        return {
            'statusCode': 500,
            'error': str(e)
        }