import * as cdk from 'aws-cdk-lib';
import * as s3 from 'aws-cdk-lib/aws-s3';
import * as lambda from 'aws-cdk-lib/aws-lambda';
import * as iam from 'aws-cdk-lib/aws-iam';
import * as stepfunctions from 'aws-cdk-lib/aws-stepfunctions';
import * as stepfunctionsTasks from 'aws-cdk-lib/aws-stepfunctions-tasks';
import * as events from 'aws-cdk-lib/aws-events';
import * as targets from 'aws-cdk-lib/aws-events-targets';
import * as sqs from 'aws-cdk-lib/aws-sqs';
import * as sns from 'aws-cdk-lib/aws-sns';
import * as ec2 from 'aws-cdk-lib/aws-ec2';
import * as rds from 'aws-cdk-lib/aws-rds';
import * as cr from 'aws-cdk-lib/custom-resources';


import * as cloudwatch from 'aws-cdk-lib/aws-cloudwatch';
import * as logs from 'aws-cdk-lib/aws-logs';
import { Construct } from 'constructs';
import * as fs from 'fs';
import * as path from 'path';

export interface SidewalkBulkManagementConfig {
  notificationType: 'SQS' | 'SNS' | 'NONE';
  createWirelessDeviceApiTps: number;
  updateWirelessDeviceApiTps: number;
  getWirelessDeviceApiTps: number;
  sqsProperties?: {
    queueName: string;
    visibilityTimeout: number;
  };
  snsProperties?: {
    topicName: string;
  };
}

export class AWSIoTWirelessDeviceBulkManagementStack extends cdk.Stack {
  constructor(scope: Construct, id: string, props?: cdk.StackProps) {
    super(scope, id, props);

    // Load configuration from external file or use defaults
    const config: SidewalkBulkManagementConfig = this.loadConfiguration();

    // Use config values directly (not CDK parameters) so they update on redeployment
    const notificationType = config.notificationType;
    const createDeviceApiTps = config.createWirelessDeviceApiTps;
    const updateDeviceApiTps = config.updateWirelessDeviceApiTps;
    const getDeviceApiTps = config.getWirelessDeviceApiTps;

    // Create boto3 layer for latest AWS SDK
    const boto3Layer = new lambda.LayerVersion(this, 'Boto3Layer', {
      code: lambda.Code.fromAsset(path.join(__dirname, '../../cdk/boto3-layer')),
      compatibleRuntimes: [lambda.Runtime.PYTHON_3_9, lambda.Runtime.PYTHON_3_10, lambda.Runtime.PYTHON_3_11, lambda.Runtime.PYTHON_3_12, lambda.Runtime.PYTHON_3_13],
      description: 'Boto3 and botocore libraries',
    });

    // S3 Buckets
    const inputBucket = new s3.Bucket(this, 'InputBucket', {
      bucketName: `aws-iot-wireless-input-${this.account}-${this.region}`,
      eventBridgeEnabled: true,
      removalPolicy: cdk.RemovalPolicy.DESTROY
    });

    const reportBucket = new s3.Bucket(this, 'ReportBucket', {
      bucketName: `aws-iot-wireless-reports-${this.account}-${this.region}`,
      removalPolicy: cdk.RemovalPolicy.DESTROY,
      lifecycleRules: [
        {
          id: 'CleanupBatchFiles',
          enabled: true,
          prefix: 'batches/',
          expiration: cdk.Duration.days(7), // Backup cleanup - batch files should be deleted by Lambda immediately
          abortIncompleteMultipartUploadAfter: cdk.Duration.days(1) // Clean up failed uploads
        }
      ]
    });


    // Minimal VPC for Aurora (Data API still requires VPC for Aurora Serverless v2)
    const vpc = new ec2.Vpc(this, 'VPC', {
      maxAzs: 2,
      natGateways: 0, // No NAT gateways needed for Data API
      subnetConfiguration: [
        {
          cidrMask: 24,
          name: 'Database',
          subnetType: ec2.SubnetType.PRIVATE_ISOLATED,
        }
      ]
    });

    // Aurora Serverless v2 cluster with Data API enabled
    // Using PostgreSQL 16 for longer support lifecycle
    const databaseCluster = new rds.DatabaseCluster(this, 'DatabaseCluster', {
      engine: rds.DatabaseClusterEngine.auroraPostgres({
        version: rds.AuroraPostgresEngineVersion.VER_16_1
      }),
      writer: rds.ClusterInstance.serverlessV2('writer'),
      vpc,
      vpcSubnets: {
        subnetType: ec2.SubnetType.PRIVATE_ISOLATED
      },
      serverlessV2MinCapacity: 0.5,
      serverlessV2MaxCapacity: 4,
      defaultDatabaseName: 'iot_devices',
      removalPolicy: cdk.RemovalPolicy.DESTROY,
      credentials: rds.Credentials.fromGeneratedSecret('dbadmin'),
      deletionProtection: false,
      backup: {
        retention: cdk.Duration.days(1)
      },
      // Enable Data API - Lambda functions access via HTTPS, not VPC
      enableDataApi: true
    });



    // Notification Services - conditionally created based on configuration
    let notificationQueue: sqs.Queue | undefined;
    let notificationTopic: sns.Topic | undefined;

    // Create resources based on the default configuration (not parameter)
    if (config.notificationType === 'SQS') {
      notificationQueue = new sqs.Queue(this, 'NotificationQueue', {
        queueName: config.sqsProperties?.queueName || 'sidewalk-bulk-notifications',
        visibilityTimeout: cdk.Duration.seconds(config.sqsProperties?.visibilityTimeout || 300),
        deadLetterQueue: {
          queue: new sqs.Queue(this, 'NotificationDLQ', {
            queueName: 'sidewalk-bulk-notifications-dlq'
          }),
          maxReceiveCount: 3
        }
      });
    }

    if (config.notificationType === 'SNS') {
      notificationTopic = new sns.Topic(this, 'NotificationTopic', {
        topicName: config.snsProperties?.topicName || 'sidewalk-bulk-notifications'
      });
    }

    // Calculate optimal batch size and concurrency based on TPS for different operations
    // Use config values directly for calculations
    // For Create operations: use createWirelessDeviceApiTps
    const createEffectiveTps = config.createWirelessDeviceApiTps;
    const createMaxConcurrency = Math.min(10, Math.max(1, createEffectiveTps));
    const createTpsPerLambda = Math.max(1, Math.floor(createEffectiveTps / createMaxConcurrency));
    const createBatchSize = Math.max(1, Math.floor(createTpsPerLambda * 300)); // 5 minutes target duration

    // For Update operations: use min(updateWirelessDeviceApiTps, getWirelessDeviceApiTps) since updates need both calls
    const updateEffectiveTps = Math.min(config.updateWirelessDeviceApiTps, config.getWirelessDeviceApiTps);
    const updateMaxConcurrency = Math.min(10, Math.max(1, updateEffectiveTps));
    const updateTpsPerLambda = Math.max(1, Math.floor(updateEffectiveTps / updateMaxConcurrency));
    const updateBatchSize = Math.max(1, Math.floor(updateTpsPerLambda * 300)); // 5 minutes target duration



    // Lambda Functions with proper environment variables
    const inputProcessingLambda = new lambda.Function(this, 'InputProcessingLambda', {
      runtime: lambda.Runtime.PYTHON_3_11,
      handler: 'input-processor/index.lambda_handler',
      code: lambda.Code.fromAsset(path.join(__dirname, '../../lambdas')),
      layers: [boto3Layer],
      timeout: cdk.Duration.minutes(10), // Increased for 100K+ device processing
      memorySize: 1024, // 1 GB for processing large JSON files and creating many batches
      environment: {
        REPORT_BUCKET: reportBucket.bucketName,
        // Create operation settings
        CREATE_BATCH_SIZE: createBatchSize.toString(),
        CREATE_MAX_CONCURRENCY: createMaxConcurrency.toString(),
        CREATE_TPS_PER_LAMBDA: createTpsPerLambda.toString(),
        // Update operation settings
        UPDATE_BATCH_SIZE: updateBatchSize.toString(),
        UPDATE_MAX_CONCURRENCY: updateMaxConcurrency.toString(),
        UPDATE_TPS_PER_LAMBDA: updateTpsPerLambda.toString(),
        NOTIFICATION_TYPE: notificationType,
        NOTIFICATION_QUEUE_URL: notificationQueue?.queueUrl || '',
        NOTIFICATION_TOPIC_ARN: notificationTopic?.topicArn || '',
        DB_CLUSTER_ARN: databaseCluster.clusterArn,
        DB_SECRET_ARN: databaseCluster.secret!.secretArn,
        DB_NAME: 'iot_devices'
      }
    });



    const apiProcessingLambda = new lambda.Function(this, 'ApiProcessingLambda', {
      runtime: lambda.Runtime.PYTHON_3_11,
      handler: 'api-processor/index.lambda_handler',
      code: lambda.Code.fromAsset(path.join(__dirname, '../../lambdas')),
      layers: [boto3Layer],
      timeout: cdk.Duration.minutes(15),
      memorySize: 512, // 512 MB for processing device batches and storing results
      environment: {
        CREATE_DEVICE_TPS: createDeviceApiTps.toString(),
        UPDATE_DEVICE_TPS: updateDeviceApiTps.toString(),
        GET_DEVICE_TPS: getDeviceApiTps.toString(),
        REPORT_BUCKET: reportBucket.bucketName,
        DB_CLUSTER_ARN: databaseCluster.clusterArn,
        DB_SECRET_ARN: databaseCluster.secret!.secretArn,
        DB_NAME: 'iot_devices'
      }
    });

    const reportNotifyLambda = new lambda.Function(this, 'ReportNotifyLambda', {
      runtime: lambda.Runtime.PYTHON_3_11,
      handler: 'report-notify/index.lambda_handler',
      code: lambda.Code.fromAsset(path.join(__dirname, '../../lambdas')),
      layers: [boto3Layer],
      timeout: cdk.Duration.minutes(10), // Increased for 100K+ device result processing
      memorySize: 2048, // 2 GB for loading many batch results and generating large reports
      environment: {
        REPORT_BUCKET: reportBucket.bucketName,
        NOTIFICATION_TYPE: notificationType,
        NOTIFICATION_QUEUE_URL: notificationQueue?.queueUrl || '',
        NOTIFICATION_TOPIC_ARN: notificationTopic?.topicArn || '',
        DB_CLUSTER_ARN: databaseCluster.clusterArn,
        DB_SECRET_ARN: databaseCluster.secret!.secretArn,
        DB_NAME: 'iot_devices'
      }
    });

    // Grant permissions
    inputBucket.grantRead(inputProcessingLambda);
    reportBucket.grantWrite(inputProcessingLambda);  // For storing batch data
    reportBucket.grantRead(apiProcessingLambda);     // For loading batch data
    reportBucket.grantWrite(apiProcessingLambda);
    reportBucket.grantRead(reportNotifyLambda);      // For loading detailed results from S3
    reportBucket.grantWrite(reportNotifyLambda);
    reportBucket.grantDelete(reportNotifyLambda);    // For cleaning up batch files



    // Grant notification permissions
    if (notificationQueue) {
      notificationQueue.grantSendMessages(reportNotifyLambda);
    }
    if (notificationTopic) {
      notificationTopic.grantPublish(reportNotifyLambda);
    }

    // Grant IoT Wireless permissions
    const iotWirelessPolicy = new iam.PolicyStatement({
      effect: iam.Effect.ALLOW,
      actions: [
        'iotwireless:CreateWirelessDevice',
        'iotwireless:UpdateWirelessDevice',
        'iotwireless:GetWirelessDevice',
        'iotwireless:ListWirelessDevices'
      ],
      resources: ['*']
    });

    // CloudWatch metrics permissions
    const cloudWatchMetricsPolicy = new iam.PolicyStatement({
      effect: iam.Effect.ALLOW,
      actions: [
        'cloudwatch:PutMetricData'
      ],
      resources: ['*']
    });

    // RDS Data API permissions
    const rdsDataApiPolicy = new iam.PolicyStatement({
      effect: iam.Effect.ALLOW,
      actions: [
        'rds-data:ExecuteStatement',
        'rds-data:BatchExecuteStatement',
        'rds-data:BeginTransaction',
        'rds-data:CommitTransaction',
        'rds-data:RollbackTransaction'
      ],
      resources: [databaseCluster.clusterArn]
    });

    // Database initialization Lambda
    const dbInitLambda = new lambda.Function(this, 'DbInitLambda', {
      runtime: lambda.Runtime.PYTHON_3_11,
      handler: 'db-init/index.lambda_handler',
      code: lambda.Code.fromAsset(path.join(__dirname, '../../lambdas')),
      layers: [boto3Layer],
      timeout: cdk.Duration.minutes(10), // Increased timeout for cluster readiness wait
      memorySize: 256, // 256 MB sufficient for database initialization
      environment: {
        DB_CLUSTER_ARN: databaseCluster.clusterArn,
        DB_SECRET_ARN: databaseCluster.secret!.secretArn,
        DB_NAME: 'iot_devices'
      }
    });

    // Database permissions - secrets and Data API
    databaseCluster.secret!.grantRead(inputProcessingLambda);
    databaseCluster.secret!.grantRead(apiProcessingLambda);
    databaseCluster.secret!.grantRead(reportNotifyLambda);
    databaseCluster.secret!.grantRead(dbInitLambda);

    inputProcessingLambda.addToRolePolicy(iotWirelessPolicy);
    inputProcessingLambda.addToRolePolicy(cloudWatchMetricsPolicy);
    inputProcessingLambda.addToRolePolicy(rdsDataApiPolicy);

    apiProcessingLambda.addToRolePolicy(iotWirelessPolicy);
    apiProcessingLambda.addToRolePolicy(cloudWatchMetricsPolicy);
    apiProcessingLambda.addToRolePolicy(rdsDataApiPolicy);

    reportNotifyLambda.addToRolePolicy(cloudWatchMetricsPolicy);
    reportNotifyLambda.addToRolePolicy(rdsDataApiPolicy);

    dbInitLambda.addToRolePolicy(rdsDataApiPolicy);

    // Create log groups with 2-week retention for all Lambda functions
    new logs.LogGroup(this, 'InputProcessingLambdaLogGroup', {
      logGroupName: `/aws/lambda/${inputProcessingLambda.functionName}`,
      retention: logs.RetentionDays.TWO_WEEKS,
      removalPolicy: cdk.RemovalPolicy.DESTROY
    });

    new logs.LogGroup(this, 'ApiProcessingLambdaLogGroup', {
      logGroupName: `/aws/lambda/${apiProcessingLambda.functionName}`,
      retention: logs.RetentionDays.TWO_WEEKS,
      removalPolicy: cdk.RemovalPolicy.DESTROY
    });

    new logs.LogGroup(this, 'ReportNotifyLambdaLogGroup', {
      logGroupName: `/aws/lambda/${reportNotifyLambda.functionName}`,
      retention: logs.RetentionDays.TWO_WEEKS,
      removalPolicy: cdk.RemovalPolicy.DESTROY
    });

    new logs.LogGroup(this, 'DbInitLambdaLogGroup', {
      logGroupName: `/aws/lambda/${dbInitLambda.functionName}`,
      retention: logs.RetentionDays.TWO_WEEKS,
      removalPolicy: cdk.RemovalPolicy.DESTROY
    });



    // Custom resource to initialize database schema
    const dbInitProvider = new cr.Provider(this, 'DbInitProvider', {
      onEventHandler: dbInitLambda,
      logRetention: logs.RetentionDays.TWO_WEEKS
    });

    const dbInitResource = new cdk.CustomResource(this, 'DbInitResource', {
      serviceToken: dbInitProvider.serviceToken,
      properties: {
        // Force update on every deployment
        timestamp: Date.now().toString(),
        // Add cluster ARN to ensure dependency
        clusterArn: databaseCluster.clusterArn
      }
    });

    // Explicit dependency - wait for cluster to be ready
    dbInitResource.node.addDependency(databaseCluster);

    // Step Functions State Machine with proper batch processing
    const validateInputState = new stepfunctionsTasks.LambdaInvoke(this, 'ValidateInputState', {
      lambdaFunction: inputProcessingLambda,
      outputPath: '$.Payload',
      retryOnServiceExceptions: true
    });




    // Create separate Map states for Create and Update operations

    // Create Operation Map State
    const createMapState = new stepfunctions.Map(this, 'CreateDevicesMap', {
      maxConcurrency: createMaxConcurrency, // Operation-specific concurrency
      itemsPath: '$.batches',
      parameters: {
        "operation.$": "$$.Map.Item.Value.operation",
        "batchName.$": "$$.Map.Item.Value.batchName",
        "tpsPerLambda.$": "$$.Map.Item.Value.tpsPerLambda",
        "deviceCount.$": "$$.Map.Item.Value.deviceCount",
        "s3Bucket.$": "$$.Map.Item.Value.s3Bucket",
        "s3Key.$": "$$.Map.Item.Value.s3Key"
      }
    });

    // Update Operation Map State  
    const updateMapState = new stepfunctions.Map(this, 'UpdateDevicesMap', {
      maxConcurrency: updateMaxConcurrency, // Operation-specific concurrency
      itemsPath: '$.batches',
      parameters: {
        "operation.$": "$$.Map.Item.Value.operation",
        "batchName.$": "$$.Map.Item.Value.batchName",
        "tpsPerLambda.$": "$$.Map.Item.Value.tpsPerLambda",
        "deviceCount.$": "$$.Map.Item.Value.deviceCount",
        "s3Bucket.$": "$$.Map.Item.Value.s3Bucket",
        "s3Key.$": "$$.Map.Item.Value.s3Key"
      }
    });

    // Create worker tasks for each operation (same Lambda, different retry configs if needed)
    const createWorkerTask = new stepfunctionsTasks.LambdaInvoke(this, 'CreateWorkerTask', {
      lambdaFunction: apiProcessingLambda,
      retryOnServiceExceptions: true
    });

    const updateWorkerTask = new stepfunctionsTasks.LambdaInvoke(this, 'UpdateWorkerTask', {
      lambdaFunction: apiProcessingLambda,
      retryOnServiceExceptions: true
    });

    // Add retry configurations for Create operations
    createWorkerTask.addRetry({
      errors: ['Lambda.ServiceException', 'Lambda.AWSLambdaException'],
      interval: cdk.Duration.seconds(2),
      maxAttempts: 3,
      backoffRate: 2.0
    });

    createWorkerTask.addRetry({
      errors: ['States.TaskFailed'],
      interval: cdk.Duration.seconds(5),
      maxAttempts: 2,
      backoffRate: 2.0
    });

    // Add retry configurations for Update operations
    updateWorkerTask.addRetry({
      errors: ['Lambda.ServiceException', 'Lambda.AWSLambdaException'],
      interval: cdk.Duration.seconds(2),
      maxAttempts: 3,
      backoffRate: 2.0
    });

    updateWorkerTask.addRetry({
      errors: ['States.TaskFailed'],
      interval: cdk.Duration.seconds(5),
      maxAttempts: 2,
      backoffRate: 2.0
    });

    // Configure the item processors
    createMapState.itemProcessor(createWorkerTask);
    updateMapState.itemProcessor(updateWorkerTask);

    const reportAndNotifyState = new stepfunctionsTasks.LambdaInvoke(this, 'ReportAndNotifyState', {
      lambdaFunction: reportNotifyLambda,
      comment: 'Generate reports and send notifications for both success and error cases',
      retryOnServiceExceptions: true
    });

    const definition = validateInputState
      .next(new stepfunctions.Choice(this, 'ValidationChoice')
        .when(
          stepfunctions.Condition.and(
            stepfunctions.Condition.isPresent('$.validationFailed'),
            stepfunctions.Condition.booleanEquals('$.validationFailed', true)
          ),
          reportAndNotifyState
        )
        .when(stepfunctions.Condition.numberEquals('$.statusCode', 400), reportAndNotifyState)
        .when(stepfunctions.Condition.numberEquals('$.statusCode', 500), reportAndNotifyState)
        .when(stepfunctions.Condition.isPresent('$.error'), reportAndNotifyState)
        .otherwise(
          new stepfunctions.Choice(this, 'OperationChoice')
            .when(stepfunctions.Condition.stringEquals('$.operation', 'create'),
              createMapState.next(reportAndNotifyState)
            )
            .when(stepfunctions.Condition.stringEquals('$.operation', 'update'),
              updateMapState.next(reportAndNotifyState)
            )
            .otherwise(reportAndNotifyState) // Fallback for unknown operations
        )
      );

    const stateMachine = new stepfunctions.StateMachine(this, 'AWSIoTWirelessDeviceBulkManagementStateMachine', {
      stateMachineName: 'AWSIoTWirelessDeviceBulkManagementStateMachine',
      definition,
      timeout: cdk.Duration.hours(4),
      logs: {
        destination: new logs.LogGroup(this, 'StateMachineLogGroup', {
          retention: logs.RetentionDays.TWO_WEEKS
        }),
        level: stepfunctions.LogLevel.ALL
      }
    });

    // S3 → EventBridge → Step Functions Trigger
    new events.Rule(this, 'S3ToStepFunctionsRule', {
      eventPattern: {
        source: ['aws.s3'],
        detailType: ['Object Created'],
        detail: {
          bucket: { name: [inputBucket.bucketName] }
        }
      },
      targets: [new targets.SfnStateMachine(stateMachine)]
    });

    // Comprehensive CloudWatch Dashboard
    this.createCombinedDashboard(stateMachine, inputProcessingLambda, apiProcessingLambda, reportNotifyLambda);

    // Outputs
    new cdk.CfnOutput(this, 'InputBucketName', {
      value: inputBucket.bucketName,
      description: 'S3 bucket for uploading device JSON files'
    });

    new cdk.CfnOutput(this, 'ReportBucketName', {
      value: reportBucket.bucketName,
      description: 'S3 bucket for processing reports'
    });

    new cdk.CfnOutput(this, 'StateMachineArn', {
      value: stateMachine.stateMachineArn,
      description: 'Step Functions state machine ARN'
    });

    if (notificationQueue) {
      new cdk.CfnOutput(this, 'NotificationQueueUrl', {
        value: notificationQueue.queueUrl,
        description: 'SQS queue URL for notifications'
      });
    }

    if (notificationTopic) {
      new cdk.CfnOutput(this, 'NotificationTopicArn', {
        value: notificationTopic.topicArn,
        description: 'SNS topic ARN for notifications'
      });
    }

    new cdk.CfnOutput(this, 'DashboardUrl', {
      value: `https://${this.region}.console.aws.amazon.com/cloudwatch/home?region=${this.region}#dashboards:name=IoT-Wireless-Bulk-Management`,
      description: 'CloudWatch Dashboard URL for IoT Wireless Bulk Management'
    });

    new cdk.CfnOutput(this, 'DatabaseClusterArn', {
      value: databaseCluster.clusterArn,
      description: 'Aurora database cluster ARN for Data API'
    });

    new cdk.CfnOutput(this, 'DatabaseSecretArn', {
      value: databaseCluster.secret!.secretArn,
      description: 'Aurora database credentials secret ARN'
    });


  }

  private createCombinedDashboard(
    stateMachine: stepfunctions.StateMachine,
    inputLambda: lambda.Function,
    apiLambda: lambda.Function,
    reportLambda: lambda.Function
  ) {
    const dashboard = new cloudwatch.Dashboard(this, 'CombinedDashboard', {
      dashboardName: 'IoT-Wireless-Bulk-Management'
    });

    // Step Functions metrics
    const stepFunctionWidget = new cloudwatch.GraphWidget({
      title: 'Step Function Executions',
      left: [
        stateMachine.metricSucceeded({ label: 'Successful Executions' }),
        stateMachine.metricFailed({ label: 'Failed Executions' }),
        stateMachine.metricStarted({ label: 'Started Executions' })
      ],
      width: 12,
      height: 6
    });

    // Lambda invocations and errors
    const lambdaInvocationsWidget = new cloudwatch.GraphWidget({
      title: 'Lambda Invocations & Errors',
      left: [
        inputLambda.metricInvocations({ label: 'Input Processor' }),
        apiLambda.metricInvocations({ label: 'API Processor' }),
        reportLambda.metricInvocations({ label: 'Report Notify' })
      ],
      right: [
        inputLambda.metricErrors({ label: 'Input Errors' }),
        apiLambda.metricErrors({ label: 'API Errors' }),
        reportLambda.metricErrors({ label: 'Report Errors' })
      ],
      width: 12,
      height: 6
    });

    // Lambda duration
    const lambdaDurationWidget = new cloudwatch.GraphWidget({
      title: 'Lambda Duration',
      left: [
        inputLambda.metricDuration({ label: 'Input Processor Duration' }),
        apiLambda.metricDuration({ label: 'API Processor Duration' }),
        reportLambda.metricDuration({ label: 'Report Notify Duration' })
      ],
      width: 12,
      height: 6
    });

    // Validation metrics
    const validationWidget = new cloudwatch.GraphWidget({
      title: 'Validation Metrics',
      left: [
        new cloudwatch.Metric({
          namespace: 'IoTWireless/BulkManagement',
          metricName: 'ValidationSuccess',
          dimensionsMap: { Operation: 'create' },
          statistic: 'Sum',
          label: 'Successful Validations (Create)'
        }),
        new cloudwatch.Metric({
          namespace: 'IoTWireless/BulkManagement',
          metricName: 'ValidationSuccess',
          dimensionsMap: { Operation: 'update' },
          statistic: 'Sum',
          label: 'Successful Validations (Update)'
        }),
        new cloudwatch.Metric({
          namespace: 'IoTWireless/BulkManagement',
          metricName: 'ValidationFailure',
          statistic: 'Sum',
          label: 'Failed Validations (All Operations)'
        })
      ],
      right: [
        new cloudwatch.Metric({
          namespace: 'IoTWireless/BulkManagement',
          metricName: 'DevicesValidated',
          dimensionsMap: { Operation: 'create' },
          statistic: 'Sum',
          label: 'Devices Validated (Create)'
        }),
        new cloudwatch.Metric({
          namespace: 'IoTWireless/BulkManagement',
          metricName: 'DevicesValidated',
          dimensionsMap: { Operation: 'update' },
          statistic: 'Sum',
          label: 'Devices Validated (Update)'
        })
      ],
      width: 12,
      height: 6
    });

    // Device processing success/failure rates
    const deviceProcessingWidget = new cloudwatch.GraphWidget({
      title: 'Device Processing Results',
      left: [
        new cloudwatch.Metric({
          namespace: 'IoTWireless/BulkManagement',
          metricName: 'DeviceCreated',
          dimensionsMap: { Operation: 'create' },
          statistic: 'Sum',
          label: 'Devices Created'
        }),
        new cloudwatch.Metric({
          namespace: 'IoTWireless/BulkManagement',
          metricName: 'DeviceUpdated',
          dimensionsMap: { Operation: 'update' },
          statistic: 'Sum',
          label: 'Devices Updated'
        })
      ],
      right: [
        new cloudwatch.Metric({
          namespace: 'IoTWireless/BulkManagement',
          metricName: 'DeviceProcessingFailure',
          dimensionsMap: { Operation: 'create' },
          statistic: 'Sum',
          label: 'Create Failures'
        }),
        new cloudwatch.Metric({
          namespace: 'IoTWireless/BulkManagement',
          metricName: 'DeviceProcessingFailure',
          dimensionsMap: { Operation: 'update' },
          statistic: 'Sum',
          label: 'Update Failures'
        })
      ],
      width: 12,
      height: 6
    });

    // API call success/failure rates
    const apiCallsWidget = new cloudwatch.GraphWidget({
      title: 'IoT Wireless API Calls',
      left: [
        new cloudwatch.Metric({
          namespace: 'IoTWireless/BulkManagement',
          metricName: 'APICallSuccess',
          dimensionsMap: { Operation: 'create', APICall: 'CreateWirelessDevice' },
          statistic: 'Sum',
          label: 'Create Device Success'
        }),
        new cloudwatch.Metric({
          namespace: 'IoTWireless/BulkManagement',
          metricName: 'APICallSuccess',
          dimensionsMap: { Operation: 'update', APICall: 'UpdateWirelessDevice' },
          statistic: 'Sum',
          label: 'Update Device Success'
        })
      ],
      right: [
        new cloudwatch.Metric({
          namespace: 'IoTWireless/BulkManagement',
          metricName: 'APICallFailure',
          dimensionsMap: { Operation: 'create' },
          statistic: 'Sum',
          label: 'Create API Failures'
        }),
        new cloudwatch.Metric({
          namespace: 'IoTWireless/BulkManagement',
          metricName: 'APICallFailure',
          dimensionsMap: { Operation: 'update' },
          statistic: 'Sum',
          label: 'Update API Failures'
        })
      ],
      width: 12,
      height: 6
    });

    // Batch processing metrics
    const batchProcessingWidget = new cloudwatch.GraphWidget({
      title: 'Batch Processing',
      left: [
        new cloudwatch.Metric({
          namespace: 'IoTWireless/BulkManagement',
          metricName: 'BatchProcessingStarted',
          dimensionsMap: { Operation: 'create' },
          statistic: 'Sum',
          label: 'Create Batches Started'
        }),
        new cloudwatch.Metric({
          namespace: 'IoTWireless/BulkManagement',
          metricName: 'BatchProcessingStarted',
          dimensionsMap: { Operation: 'update' },
          statistic: 'Sum',
          label: 'Update Batches Started'
        }),
        new cloudwatch.Metric({
          namespace: 'IoTWireless/BulkManagement',
          metricName: 'BatchProcessingCompleted',
          dimensionsMap: { Operation: 'create' },
          statistic: 'Sum',
          label: 'Create Batches Completed'
        }),
        new cloudwatch.Metric({
          namespace: 'IoTWireless/BulkManagement',
          metricName: 'BatchProcessingCompleted',
          dimensionsMap: { Operation: 'update' },
          statistic: 'Sum',
          label: 'Update Batches Completed'
        })
      ],
      right: [
        new cloudwatch.Metric({
          namespace: 'IoTWireless/BulkManagement',
          metricName: 'BatchSuccessRate',
          dimensionsMap: { Operation: 'create' },
          statistic: 'Average',
          label: 'Create Success Rate (%)'
        }),
        new cloudwatch.Metric({
          namespace: 'IoTWireless/BulkManagement',
          metricName: 'BatchSuccessRate',
          dimensionsMap: { Operation: 'update' },
          statistic: 'Average',
          label: 'Update Success Rate (%)'
        })
      ],
      width: 12,
      height: 6
    });

    // Error breakdown by type
    const errorBreakdownWidget = new cloudwatch.GraphWidget({
      title: 'Error Breakdown by Type',
      left: [
        new cloudwatch.Metric({
          namespace: 'IoTWireless/BulkManagement',
          metricName: 'DeviceProcessingFailure',
          dimensionsMap: { Operation: 'create', ErrorCode: 'ValidationException' },
          statistic: 'Sum',
          label: 'Create Validation Errors'
        }),
        new cloudwatch.Metric({
          namespace: 'IoTWireless/BulkManagement',
          metricName: 'DeviceProcessingFailure',
          dimensionsMap: { Operation: 'create', ErrorCode: 'ThrottlingException' },
          statistic: 'Sum',
          label: 'Create Throttling Errors'
        }),
        new cloudwatch.Metric({
          namespace: 'IoTWireless/BulkManagement',
          metricName: 'DeviceProcessingFailure',
          dimensionsMap: { Operation: 'create', ErrorCode: 'ConflictException' },
          statistic: 'Sum',
          label: 'Create Conflict Errors'
        })
      ],
      right: [
        new cloudwatch.Metric({
          namespace: 'IoTWireless/BulkManagement',
          metricName: 'DeviceProcessingFailure',
          dimensionsMap: { Operation: 'update', ErrorCode: 'ValidationException' },
          statistic: 'Sum',
          label: 'Update Validation Errors'
        }),
        new cloudwatch.Metric({
          namespace: 'IoTWireless/BulkManagement',
          metricName: 'DeviceProcessingFailure',
          dimensionsMap: { Operation: 'update', ErrorCode: 'ThrottlingException' },
          statistic: 'Sum',
          label: 'Update Throttling Errors'
        }),
        new cloudwatch.Metric({
          namespace: 'IoTWireless/BulkManagement',
          metricName: 'DeviceProcessingFailure',
          dimensionsMap: { Operation: 'update', ErrorCode: 'ConflictException' },
          statistic: 'Sum',
          label: 'Update Conflict Errors'
        })
      ],
      width: 12,
      height: 6
    });

    dashboard.addWidgets(
      stepFunctionWidget,
      lambdaInvocationsWidget,
      lambdaDurationWidget,
      validationWidget,
      deviceProcessingWidget,
      apiCallsWidget,
      batchProcessingWidget,
      errorBreakdownWidget
    );
  }

  private loadConfiguration(): SidewalkBulkManagementConfig {
    // Try to load from config.json in the current directory
    const configPath = path.resolve('config.json');

    try {
      if (fs.existsSync(configPath)) {
        const configContent = fs.readFileSync(configPath, 'utf8');
        return JSON.parse(configContent) as SidewalkBulkManagementConfig;
      }
    } catch (error) {
      console.warn(`Warning: Could not load config.json: ${error}. Using defaults.`);
    }

    // Return default configuration
    return {
      notificationType: 'SQS',
      createWirelessDeviceApiTps: 10,
      updateWirelessDeviceApiTps: 10,
      getWirelessDeviceApiTps: 10,
      sqsProperties: {
        queueName: 'sidewalk-bulk-notifications',
        visibilityTimeout: 300
      }
    };
  }
}
