"""
Report and Notify Lambda - Generates comprehensive reports and sends notifications using RDS Data API
"""
import json
import boto3
import logging
import os
import uuid
from datetime import datetime
from typing import Dict, Any, List

logger = logging.getLogger()
logger.setLevel(logging.INFO)


# Initialize AWS clients
s3_client = boto3.client('s3')
sns_client = boto3.client('sns')
cloudwatch = boto3.client('cloudwatch')
rds_client = boto3.client('rds-data')

def execute_sql(sql, parameters=None):
    """Execute SQL using RDS Data API"""
    try:
        request = {
            'resourceArn': os.environ['DB_CLUSTER_ARN'],
            'secretArn': os.environ['DB_SECRET_ARN'],
            'database': os.environ['DB_NAME'],
            'sql': sql
        }
        
        if parameters:
            request['parameters'] = parameters
            
        response = rds_client.execute_statement(**request)
        return response
        
    except Exception as e:
        logger.error(f"Failed to execute SQL: {str(e)}")
        logger.error(f"SQL: {sql}")
        raise

def cleanup_batch_result_files(results: List[Dict], s3_bucket: str):
    """Clean up batch result files from S3"""
    try:
        result_keys_to_delete = []
        
        for result in results:
            if result.get('results_s3_key'):
                result_keys_to_delete.append({'Key': result['results_s3_key']})
        
        if result_keys_to_delete:
            logger.info(f"Cleaning up {len(result_keys_to_delete)} batch result files")
            
            # Delete result files
            s3_client.delete_objects(
                Bucket=s3_bucket,
                Delete={'Objects': result_keys_to_delete}
            )
            
            logger.info(f"Successfully cleaned up {len(result_keys_to_delete)} batch result files")
        else:
            logger.info("No batch result files to clean up")
            
    except Exception as e:
        logger.error(f"Failed to cleanup batch result files: {str(e)}")

def cleanup_batch_files(batch_name: str, s3_bucket: str):
    """
    Clean up temporary batch files from S3 after Step Functions execution
    """
    try:
        # Handle both main batch name and individual batch names
        # If batch_name ends with -1, -2, etc., extract the main batch name
        main_batch_name = batch_name
        if batch_name and '-' in batch_name:
            # Check if it ends with -number (individual batch name)
            parts = batch_name.rsplit('-', 1)
            if len(parts) == 2 and parts[1].isdigit():
                main_batch_name = parts[0]
                logger.info(f"Extracted main batch name: {main_batch_name} from individual batch: {batch_name}")
        
        # List all batch files for this batch
        batch_prefix = f"batches/{main_batch_name}/"
        
        logger.info(f"Starting cleanup for batch: {main_batch_name}")
        logger.info(f"Looking for batch files with prefix: {batch_prefix} in bucket: {s3_bucket}")
        
        # Use paginator to handle large numbers of batch files
        paginator = s3_client.get_paginator('list_objects_v2')
        page_iterator = paginator.paginate(
            Bucket=s3_bucket,
            Prefix=batch_prefix
        )
        
        objects_to_delete = []
        for page in page_iterator:
            if 'Contents' in page:
                for obj in page['Contents']:
                    objects_to_delete.append({'Key': obj['Key']})
        
        if not objects_to_delete:
            logger.warning(f"No batch files found to cleanup for batch: {main_batch_name}")
            logger.warning(f"Expected files with prefix: {batch_prefix}")
            
            # List what's actually in the batches/ folder for debugging
            debug_response = s3_client.list_objects_v2(
                Bucket=s3_bucket,
                Prefix='batches/',
                MaxKeys=10
            )
            if 'Contents' in debug_response:
                logger.info(f"Found {len(debug_response['Contents'])} files in batches/ folder:")
                for obj in debug_response['Contents']:
                    logger.info(f"  - {obj['Key']}")
            else:
                logger.info("No files found in batches/ folder at all")
            return
        
        # Log which files are being deleted
        logger.info(f"Found {len(objects_to_delete)} batch files to delete:")
        for obj in objects_to_delete[:10]:  # Log first 10 files
            logger.info(f"  - {obj['Key']}")
        if len(objects_to_delete) > 10:
            logger.info(f"  ... and {len(objects_to_delete) - 10} more files")
        
        # Delete files in batches of 1000 (S3 limit)
        deleted_count = 0
        for i in range(0, len(objects_to_delete), 1000):
            batch_to_delete = objects_to_delete[i:i + 1000]
            
            delete_response = s3_client.delete_objects(
                Bucket=s3_bucket,
                Delete={'Objects': batch_to_delete}
            )
            
            # Check for errors in deletion
            if 'Errors' in delete_response and delete_response['Errors']:
                logger.error(f"Errors during batch file deletion:")
                for error in delete_response['Errors']:
                    logger.error(f"  - {error['Key']}: {error['Code']} - {error['Message']}")
            
            if 'Deleted' in delete_response:
                deleted_count += len(delete_response['Deleted'])
        
        logger.info(f"Successfully cleaned up {deleted_count} batch files for batch: {main_batch_name}")
        
        # Emit cleanup metric
        cloudwatch.put_metric_data(
            Namespace='IoTWireless/BulkManagement',
            MetricData=[{
                'MetricName': 'BatchFilesCleanedUp',
                'Value': deleted_count,
                'Unit': 'Count',
                'Timestamp': datetime.utcnow(),
                'Dimensions': [
                    {'Name': 'BatchName', 'Value': main_batch_name}
                ]
            }]
        )
        
    except Exception as e:
        logger.error(f"Failed to cleanup batch files for {batch_name}: {str(e)}")
        logger.error(f"Error details: {type(e).__name__}: {str(e)}")
        # Don't raise exception - cleanup failure shouldn't break the main workflow

def get_batch_info_from_database(batch_id: str) -> Dict[str, Any]:
    """Get batch information from database using RDS Data API"""
    try:
        sql = """
            SELECT batch_id, operation, total_devices, s3_url, status, created_at
            FROM batches 
            WHERE batch_id = :batch_id
        """
        
        parameters = [
            {'name': 'batch_id', 'value': {'stringValue': batch_id}}
        ]
        
        response = execute_sql(sql, parameters)
        
        if response.get('records') and len(response['records']) > 0:
            record = response['records'][0]
            return {
                'batch_id': record[0].get('stringValue', ''),
                'operation': record[1].get('stringValue', ''),
                'total_devices': record[2].get('longValue', 0),
                's3_url': record[3].get('stringValue', ''),
                'status': record[4].get('stringValue', ''),
                'created_at': record[5].get('stringValue', '')
            }
    except Exception as e:
        logger.error(f"Failed to get batch info from database: {str(e)}")
    
    return {}

def update_batch_completion(batch_id: str, successful_count: int, failed_count: int):
    """Update batch completion status in database using RDS Data API"""
    try:
        sql = """
            UPDATE batches 
            SET status = 'COMPLETED',
                processed_devices = :processed_devices,
                successful_devices = :successful_devices,
                failed_devices = :failed_devices,
                completed_at = CURRENT_TIMESTAMP
            WHERE batch_id = :batch_id
        """
        
        parameters = [
            {'name': 'processed_devices', 'value': {'longValue': successful_count + failed_count}},
            {'name': 'successful_devices', 'value': {'longValue': successful_count}},
            {'name': 'failed_devices', 'value': {'longValue': failed_count}},
            {'name': 'batch_id', 'value': {'stringValue': batch_id}}
        ]
        
        execute_sql(sql, parameters)
        logger.info(f"Updated batch {batch_id} completion status")
    except Exception as e:
        logger.error(f"Failed to update batch completion: {str(e)}")

def emit_metric(metric_name: str, value: float, unit: str = 'Count', dimensions: Dict[str, str] = None):
    """Emit CloudWatch metric"""
    try:
        metric_data = {
            'MetricName': metric_name,
            'Value': value,
            'Unit': unit,
            'Timestamp': datetime.utcnow()
        }
        
        if dimensions:
            metric_data['Dimensions'] = [
                {'Name': k, 'Value': v} for k, v in dimensions.items()
            ]
        
        cloudwatch.put_metric_data(
            Namespace='IoTWireless/BulkManagement',
            MetricData=[metric_data]
        )
        
    except Exception as e:
        logger.error(f"Failed to emit metric {metric_name}: {str(e)}")

def load_detailed_results_from_s3(results: List[Dict]) -> List[Dict]:
    """Load detailed results from S3 for batches that stored them there"""
    s3_keys_to_load = []
    
    # Collect all S3 keys to load
    for i, result in enumerate(results):
        if result.get('results_s3_key') and result.get('results_s3_bucket'):
            s3_keys_to_load.append((i, result['results_s3_key'], result['results_s3_bucket']))
    
    logger.info(f"Found {len(s3_keys_to_load)} batches with S3 results to load")
    
    if not s3_keys_to_load:
        logger.warning("No S3 result keys found - returning original results")
        return results
    
    # Try to use concurrent loading, fall back to sequential if there are issues
    try:
        import concurrent.futures
        use_concurrent = len(s3_keys_to_load) > 5  # Only use concurrent for more than 5 batches
    except ImportError:
        logger.warning("concurrent.futures not available - using sequential loading")
        use_concurrent = False
    
    def load_single_result(index, s3_key, s3_bucket, original_result):
        try:
            response = s3_client.get_object(Bucket=s3_bucket, Key=s3_key)
            detailed_data = json.loads(response['Body'].read().decode('utf-8'))
            
            successful_devices = detailed_data.get('successful_devices', [])
            failed_devices = detailed_data.get('failed_devices', [])
            
            logger.info(f"Loaded batch {index} from S3: {len(successful_devices)} successful, {len(failed_devices)} failed devices")
            
            # Debug: Log first few devices to verify structure
            if successful_devices:
                logger.info(f"Sample successful device: {successful_devices[0]}")
            if failed_devices:
                logger.info(f"Sample failed device: {failed_devices[0]}")
            
            # Merge the detailed data with the summary
            merged_result = {
                **original_result,  # Keep the summary data
                'successful_devices': successful_devices,
                'failed_devices': failed_devices
            }
            
            return (index, merged_result)
            
        except Exception as e:
            logger.error(f"Failed to load detailed results from S3 {s3_key}: {str(e)}")
            # Fall back to using summary data only
            return (index, {
                **original_result,
                'successful_devices': [],
                'failed_devices': [],
                'note': f'Detailed results could not be loaded from S3: {str(e)}'
            })
    
    loaded_results = {}
    
    if use_concurrent:
        # Load S3 results concurrently (max 10 concurrent to avoid overwhelming S3)
        logger.info("Using concurrent S3 loading")
        with concurrent.futures.ThreadPoolExecutor(max_workers=10) as executor:
            future_to_index = {
                executor.submit(load_single_result, i, s3_key, s3_bucket, results[i]): i 
                for i, s3_key, s3_bucket in s3_keys_to_load
            }
            
            for future in concurrent.futures.as_completed(future_to_index):
                try:
                    index, result = future.result(timeout=30)  # 30 second timeout per S3 load
                    loaded_results[index] = result
                except Exception as e:
                    index = future_to_index[future]
                    logger.error(f"Failed to load result for batch {index}: {str(e)}")
                    loaded_results[index] = {
                        **results[index],
                        'successful_devices': [],
                        'failed_devices': [],
                        'note': f'S3 load timeout or error: {str(e)}'
                    }
    else:
        # Load S3 results sequentially
        logger.info("Using sequential S3 loading")
        for i, s3_key, s3_bucket in s3_keys_to_load:
            try:
                index, result = load_single_result(i, s3_key, s3_bucket, results[i])
                loaded_results[index] = result
            except Exception as e:
                logger.error(f"Failed to load result for batch {i}: {str(e)}")
                loaded_results[i] = {
                    **results[i],
                    'successful_devices': [],
                    'failed_devices': [],
                    'note': f'S3 load error: {str(e)}'
                }
    
    # Reconstruct results in original order
    final_results = []
    for i in range(len(results)):
        if i in loaded_results:
            final_results.append(loaded_results[i])
        else:
            # Use original result if not loaded from S3
            final_results.append(results[i])
    
    logger.info(f"Successfully loaded detailed results for {len(loaded_results)} batches out of {len(s3_keys_to_load)} S3 batches")
    return final_results

def get_batch_summary_from_results(results: List[Dict], batch_id: str) -> Dict[str, Any]:
    """Generate batch summary from processing results"""
    successful_devices = 0
    failed_devices = 0
    error_breakdown = {}
    
    for result in results:
        if result.get('statusCode') == 200:
            # Use new summary fields if available, fall back to counting arrays
            if 'success_count' in result and 'failure_count' in result:
                successful_devices += result.get('success_count', 0)
                failed_devices += result.get('failure_count', 0)
            else:
                # Legacy format - count arrays
                successful_devices += len(result.get('successful_devices', []))
                failed_devices += len(result.get('failed_devices', []))
        else:
            # Entire batch failed
            failed_devices += result.get('total_devices', 0)
        
        # Count error types from detailed device data
        for error in result.get('failed_devices', []):
            error_code = error.get('error_code', 'Unknown')
            error_breakdown[error_code] = error_breakdown.get(error_code, 0) + 1
    
    return {
        'batch_id': batch_id,
        'total_devices': successful_devices + failed_devices,
        'successful_devices': successful_devices,
        'failed_devices': failed_devices,
        'success_rate': (successful_devices / (successful_devices + failed_devices) * 100) if (successful_devices + failed_devices) > 0 else 0,
        'error_breakdown': error_breakdown,
        'timestamp': datetime.utcnow().isoformat()
    }

def generate_detailed_report(batch_summary: Dict[str, Any], results: List[Dict]) -> str:
    """Generate detailed JSON report"""
    detailed_report = {
        'batch_summary': batch_summary,
        'devices': []
    }
    
    logger.info(f"Generating detailed report from {len(results)} result batches")
    
    for i, result in enumerate(results):
        successful_devices = result.get('successful_devices', [])
        failed_devices = result.get('failed_devices', [])
        
        logger.info(f"Batch {i}: {len(successful_devices)} successful, {len(failed_devices)} failed devices")
        
        # Add successful devices
        for device in successful_devices:
            detailed_report['devices'].append({
                'smsn': device.get('smsn', ''),
                'aws_wireless_device_id': device.get('aws_wireless_device_id', ''),
                'status': 'SUCCESS',
                'error_code': None,
                'error_message': None,
                'timestamp': device.get('timestamp', ''),
                'retryable': None
            })
        
        # Add failed devices
        for device in failed_devices:
            detailed_report['devices'].append({
                'smsn': device.get('smsn', ''),
                'aws_wireless_device_id': device.get('aws_wireless_device_id', ''),
                'status': 'FAILED',
                'error_code': device.get('error_code', ''),
                'error_message': device.get('error_message', ''),
                'timestamp': device.get('timestamp', ''),
                'retryable': device.get('retryable', False)
            })
    
    logger.info(f"Generated detailed report with {len(detailed_report['devices'])} total device entries")
    
    return json.dumps(detailed_report, indent=2)

def generate_summary_report(batch_summary: Dict[str, Any]) -> str:
    """Generate summary JSON report"""
    return json.dumps(batch_summary, indent=2)

def handle_validation_failure(event: Dict[str, Any]) -> Dict[str, Any]:
    """Handle validation failure case - generate reports and send notifications"""
    logger.info("Handling validation failure")
    logger.info(f"Validation failure event: {json.dumps(event)}")
    
    batch_name = event.get('batchName')
    operation = event.get('operation', 'unknown')
    validation_summary = event.get('validationSummary', {})
    validation_errors_s3_key = event.get('validationErrorsS3Key')
    
    logger.info(f"Batch name: {batch_name}")
    logger.info(f"Validation errors S3 key: {validation_errors_s3_key}")
    
    # Load validation error details from S3
    report_bucket = os.environ.get('REPORT_BUCKET')
    validation_error_data = {}
    
    if validation_errors_s3_key:
        try:
            logger.info(f"Attempting to load from s3://{report_bucket}/{validation_errors_s3_key}")
            response = s3_client.get_object(Bucket=report_bucket, Key=validation_errors_s3_key)
            validation_error_data = json.loads(response['Body'].read().decode('utf-8'))
            logger.info(f"Successfully loaded validation error data with {len(validation_error_data.get('invalid_devices', []))} invalid devices")
        except s3_client.exceptions.NoSuchKey:
            logger.error(f"Validation errors file not found at s3://{report_bucket}/{validation_errors_s3_key}")
        except Exception as e:
            logger.error(f"Failed to load validation error data: {str(e)}")
            logger.error(f"Error type: {type(e).__name__}")
    else:
        logger.warning("No validationErrorsS3Key provided in event")
    
    # Use validation_error_data if loaded, otherwise fall back to validation_summary
    if not validation_error_data and validation_summary:
        logger.warning("Using validation_summary from event as fallback")
        validation_error_data = {
            'timestamp': datetime.utcnow().isoformat(),
            'invalid_devices': []  # Will be empty, but at least we can generate a report
        }
    
    # Create batch summary for validation failure
    batch_summary = {
        'batch_id': batch_name,
        'operation': operation,
        'status': 'VALIDATION_FAILED',
        'total_devices': validation_summary.get('total_devices', 0),
        'successful_devices': 0,
        'failed_devices': 0,
        'validation_failures': validation_summary.get('invalid_devices', 0),
        'success_rate': 0.0,
        'start_time': validation_error_data.get('timestamp', datetime.utcnow().isoformat()),
        'end_time': datetime.utcnow().isoformat(),
        's3_input_url': event.get('s3Url', ''),
        'validation_errors': validation_summary.get('validation_errors', [])
    }
    
    # Generate detailed report with validation failures
    detailed_report = {
        'batch_summary': batch_summary,
        'devices': []
    }
    
    # Add all invalid devices to the report
    invalid_devices = validation_error_data.get('invalid_devices', [])
    logger.info(f"Processing {len(invalid_devices)} invalid devices for report")
    for invalid_device in invalid_devices:
        device_data = invalid_device.get('device', {})
        errors = invalid_device.get('errors', [])
        detailed_report['devices'].append({
            'smsn': device_data.get('smsn', ''),
            'device_name': device_data.get('deviceName') or device_data.get('device_name', ''),
            'status': 'VALIDATION_FAILED',
            'error_code': 'VALIDATION_ERROR',
            'error_message': '; '.join(errors),
            'timestamp': batch_summary['start_time'],
            'retryable': False
        })
    
    # Upload reports to S3
    timestamp = datetime.utcnow().strftime('%Y%m%d_%H%M%S')
    summary_key = f"reports/summary_{batch_name}_{timestamp}.json"
    detailed_key = f"reports/detailed_{batch_name}_{timestamp}.json"
    
    summary_report = generate_summary_report(batch_summary)
    detailed_report_json = json.dumps(detailed_report, indent=2)
    
    summary_url = upload_report_to_s3(summary_report, summary_key, 'application/json')
    detailed_url = upload_report_to_s3(detailed_report_json, detailed_key, 'application/json')
    
    report_urls = {
        'summary': summary_url,
        'detailed': detailed_url
    }
    
    # Update database
    update_batch_completion(batch_name, 0, 0, 'VALIDATION_FAILED')
    
    # Emit metrics
    emit_metric('ValidationFailure', 1, 'Count', {'Operation': operation})
    emit_metric('DevicesValidationFailed', validation_summary.get('invalid_devices', 0), 'Count', {'Operation': operation})
    
    # Send notifications
    notification_type = os.environ.get('NOTIFICATION_TYPE', 'NONE')
    
    if notification_type == 'SQS':
        send_sqs_notification(batch_summary, report_urls)
    elif notification_type == 'SNS':
        send_sns_notification(batch_summary, report_urls)
    
    logger.info(f"Validation failure processing completed for batch {batch_name}")
    
    return {
        'statusCode': 200,
        'batch_id': batch_name,
        'status': 'VALIDATION_FAILED',
        'report_urls': report_urls
    }

def upload_report_to_s3(report_content: str, report_key: str, content_type: str = 'text/plain'):
    """Upload report to S3"""
    try:
        bucket_name = os.environ['REPORT_BUCKET']
        
        s3_client.put_object(
            Bucket=bucket_name,
            Key=report_key,
            Body=report_content,
            ContentType=content_type
        )
        
        s3_url = f"s3://{bucket_name}/{report_key}"
        logger.info(f"Report uploaded to {s3_url}")
        return s3_url
        
    except Exception as e:
        logger.error(f"Failed to upload report to S3: {str(e)}")
        return None

def send_sqs_notification(batch_summary: Dict[str, Any], report_urls: Dict[str, str]):
    """Send notification via SQS"""
    try:
        queue_url = os.environ.get('NOTIFICATION_QUEUE_URL')
        if not queue_url:
            logger.warning("SQS queue URL not configured")
            return
        
        sqs_client = boto3.client('sqs')
        
        message = {
            'notification_type': 'batch_completion',
            'batch_summary': batch_summary,
            'report_urls': report_urls,
            'timestamp': datetime.utcnow().isoformat()
        }
        
        sqs_client.send_message(
            QueueUrl=queue_url,
            MessageBody=json.dumps(message)
        )
        
        logger.info(f"SQS notification sent for batch {batch_summary['batch_id']}")
        
    except Exception as e:
        logger.error(f"Failed to send SQS notification: {str(e)}")

def send_sns_notification(batch_summary: Dict[str, Any], report_urls: Dict[str, str]):
    """Send notification via SNS"""
    try:
        topic_arn = os.environ.get('NOTIFICATION_TOPIC_ARN')
        if not topic_arn:
            logger.warning("SNS topic ARN not configured")
            return
        
        subject = f"IoT Device Batch Processing Complete - {batch_summary['batch_id']}"
        
        message = f"""
Batch Processing Complete

Batch ID: {batch_summary['batch_id']}
Total Devices: {batch_summary['total_devices']}
Successful: {batch_summary['successful_devices']}
Failed: {batch_summary['failed_devices']}
Success Rate: {batch_summary['success_rate']:.1f}%

Reports:
- Summary: {report_urls.get('summary', 'N/A')}
- Detailed: {report_urls.get('detailed', 'N/A')}

Timestamp: {batch_summary['timestamp']}
        """
        
        sns_client.publish(
            TopicArn=topic_arn,
            Subject=subject,
            Message=message
        )
        
        logger.info(f"SNS notification sent for batch {batch_summary['batch_id']}")
        
    except Exception as e:
        logger.error(f"Failed to send SNS notification: {str(e)}")

def lambda_handler(event, context):
    """
    Main handler for report generation and notifications
    """
    logger.info(f"Report and notify handler started")
    logger.info(f"Event type: {type(event)}")
    logger.info(f"Event: {json.dumps(event)}")
    
    try:
        # Handle different event structures from Step Functions
        if isinstance(event, list):
            # Direct list of results from Map state
            results = []
            batch_id = str(uuid.uuid4())
            operation = 'unknown'
            main_batch_name = None
            
            for item in event:
                if isinstance(item, dict):
                    # Extract payload from Step Functions response
                    if 'Payload' in item and isinstance(item['Payload'], dict):
                        payload = item['Payload']
                        results.append(payload)
                        # Get batch info from first payload
                        if not batch_id or batch_id == str(uuid.uuid4()):
                            individual_batch_name = payload.get('batch_name') or payload.get('batchName') or str(uuid.uuid4())
                            batch_id = individual_batch_name
                            operation = payload.get('operation', 'unknown')
                            
                            # Extract main batch name from individual batch name
                            if individual_batch_name and '-' in individual_batch_name:
                                parts = individual_batch_name.rsplit('-', 1)
                                if len(parts) == 2 and parts[1].isdigit():
                                    main_batch_name = parts[0]
                                    logger.info(f"Extracted main batch name: {main_batch_name} from individual batch: {individual_batch_name}")
                                else:
                                    main_batch_name = individual_batch_name
                            else:
                                main_batch_name = individual_batch_name
                    else:
                        results.append(item)
                else:
                    results.append(item)
        else:
            # Single event object
            batch_id = event.get('batchName') or event.get('batch_id') or str(uuid.uuid4())
            operation = event.get('operation', 'unknown')
            main_batch_name = batch_id
            
            # Check if this is a validation failure case
            if event.get('validationFailed'):
                logger.info(f"Processing validation failure for batch {batch_id}")
                return handle_validation_failure(event)
            
            # Get results from Step Functions execution
            if 'results' in event:
                results = event['results']
            elif 'Payload' in event:
                results = [event['Payload']]
            else:
                results = [event]
        
        logger.info(f"Processing batch {batch_id} with {len(results)} results")
        logger.info(f"Results count: {len(results)}")
        
        # Debug: Log the structure of the first result to understand the format
        if results:
            logger.info(f"First result structure: {json.dumps(results[0], indent=2)}")
            logger.info(f"First result keys: {list(results[0].keys()) if isinstance(results[0], dict) else 'Not a dict'}")
        
        # Load detailed results from S3 if they were stored there
        detailed_results = load_detailed_results_from_s3(results)
        
        # Debug: Log the structure after S3 loading
        if detailed_results:
            logger.info(f"After S3 loading - first detailed result keys: {list(detailed_results[0].keys()) if isinstance(detailed_results[0], dict) else 'Not a dict'}")
            if 'successful_devices' in detailed_results[0]:
                logger.info(f"First detailed result has {len(detailed_results[0]['successful_devices'])} successful devices")
            if 'failed_devices' in detailed_results[0]:
                logger.info(f"First detailed result has {len(detailed_results[0]['failed_devices'])} failed devices")
        
        # Get batch information from database
        batch_info = get_batch_info_from_database(batch_id)
        
        # Generate batch summary from results (using detailed results for error breakdown)
        batch_summary = get_batch_summary_from_results(detailed_results, batch_id)
        
        # Add batch info from database to summary
        if batch_info:
            batch_summary.update({
                'operation': batch_info.get('operation', operation),
                's3_input_url': batch_info.get('s3_url', ''),
                'created_at': batch_info.get('created_at', '')
            })
        
        # Update batch completion in database
        update_batch_completion(batch_id, batch_summary['successful_devices'], batch_summary['failed_devices'])
        
        # Generate reports using detailed results
        summary_report = generate_summary_report(batch_summary)
        detailed_report = generate_detailed_report(batch_summary, detailed_results)
        
        # Upload reports to S3 - all reports in single folder
        timestamp = datetime.utcnow().strftime('%Y%m%d_%H%M%S')
        
        summary_key = f"reports/summary_{batch_id}_{timestamp}.json"
        detailed_key = f"reports/detailed_{batch_id}_{timestamp}.json"
        
        summary_url = upload_report_to_s3(summary_report, summary_key, 'application/json')
        detailed_url = upload_report_to_s3(detailed_report, detailed_key, 'application/json')
        
        report_urls = {
            'summary': summary_url,
            'detailed': detailed_url
        }
        
        # Emit CloudWatch metrics
        emit_metric('BatchCompleted', 1, 'Count', {'Operation': operation})
        emit_metric('DevicesProcessed', batch_summary['total_devices'], 'Count', {'Operation': operation})
        emit_metric('DevicesSuccessful', batch_summary['successful_devices'], 'Count', {'Operation': operation})
        emit_metric('DevicesFailed', batch_summary['failed_devices'], 'Count', {'Operation': operation})
        emit_metric('BatchSuccessRate', batch_summary['success_rate'], 'Percent', {'Operation': operation})
        
        # Send notifications based on configuration
        notification_type = os.environ.get('NOTIFICATION_TYPE', 'NONE')
        
        if notification_type == 'SQS':
            send_sqs_notification(batch_summary, report_urls)
        elif notification_type == 'SNS':
            send_sns_notification(batch_summary, report_urls)
        
        # Clean up temporary files from S3
        report_bucket = os.environ.get('REPORT_BUCKET')
        if report_bucket:
            # Use main_batch_name if available, otherwise fall back to batch_id
            cleanup_batch_name = main_batch_name if 'main_batch_name' in locals() and main_batch_name else batch_id
            logger.info(f"Cleaning up batch files for: {cleanup_batch_name}")
            cleanup_batch_files(cleanup_batch_name, report_bucket)
            
            # Also clean up batch result files
            cleanup_batch_result_files(detailed_results, report_bucket)
        else:
            logger.warning(f"REPORT_BUCKET environment variable not set - cannot cleanup batch files for: {batch_id}")
        
        logger.info(f"Report and notification processing completed for batch {batch_id}")
        
        return {
            'statusCode': 200,
            'batch_id': batch_id,
            'batch_summary': batch_summary,
            'report_urls': report_urls
        }
        
    except Exception as e:
        logger.error(f"Report and notify processing failed: {str(e)}")
        return {
            'statusCode': 500,
            'error': str(e)
        }