"""
Database Initialization Lambda - Creates database schema for IoT device management using RDS Data API
"""
import json
import boto3
import logging
import os
from datetime import datetime

logger = logging.getLogger()
logger.setLevel(logging.INFO)


def get_rds_data_client():
    """Get RDS Data API client"""
    return boto3.client('rds-data')

def execute_sql(rds_client, sql, parameters=None):
    """Execute SQL using RDS Data API"""
    try:
        request = {
            'resourceArn': os.environ['DB_CLUSTER_ARN'],
            'secretArn': os.environ['DB_SECRET_ARN'],
            'database': os.environ['DB_NAME'],
            'sql': sql
        }
        
        if parameters:
            request['parameters'] = parameters
            
        response = rds_client.execute_statement(**request)
        logger.info(f"SQL executed successfully: {sql[:100]}...")
        return response
        
    except Exception as e:
        logger.error(f"Failed to execute SQL: {str(e)}")
        logger.error(f"SQL: {sql}")
        raise

def create_database_schema(rds_client):
    """Create the database schema using RDS Data API"""
    
    # Split schema creation into individual statements for Data API
    schema_statements = [
        """
        CREATE TABLE IF NOT EXISTS devices (
            smsn VARCHAR(100) PRIMARY KEY,
            aws_wireless_device_id VARCHAR(36) UNIQUE,
            device_name VARCHAR(255) NOT NULL,
            device_profile_id VARCHAR(36) NOT NULL,
            uplink_destination_name VARCHAR(255),
            batch_id VARCHAR(100),
            created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
            status VARCHAR(50) NOT NULL DEFAULT 'PENDING',
            status_details TEXT,
            positioning_enabled BOOLEAN DEFAULT FALSE,
            positioning_destination_name VARCHAR(255)
        )
        """,
        "CREATE INDEX IF NOT EXISTS idx_status ON devices(status)",
        "CREATE INDEX IF NOT EXISTS idx_aws_wireless_device_id ON devices(aws_wireless_device_id)",
        "CREATE INDEX IF NOT EXISTS idx_batch_id ON devices(batch_id)",
        "CREATE INDEX IF NOT EXISTS idx_created_at ON devices(created_at)",
        "CREATE INDEX IF NOT EXISTS idx_updated_at ON devices(updated_at)",
        """
        CREATE TABLE IF NOT EXISTS batches (
            batch_id VARCHAR(100) PRIMARY KEY,
            operation VARCHAR(20) NOT NULL,
            total_devices INTEGER NOT NULL DEFAULT 0,
            processed_devices INTEGER NOT NULL DEFAULT 0,
            successful_devices INTEGER NOT NULL DEFAULT 0,
            failed_devices INTEGER NOT NULL DEFAULT 0,
            status VARCHAR(50) NOT NULL DEFAULT 'PROCESSING',
            s3_url VARCHAR(500),
            created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
            completed_at TIMESTAMP
        )
        """,
        "CREATE INDEX IF NOT EXISTS idx_batch_status ON batches(status)",
        "CREATE INDEX IF NOT EXISTS idx_batch_created_at ON batches(created_at)",
        """
        CREATE OR REPLACE FUNCTION update_updated_at_column()
        RETURNS TRIGGER AS $
        BEGIN
            NEW.updated_at = CURRENT_TIMESTAMP;
            RETURN NEW;
        END;
        $ language 'plpgsql'
        """,
        "DROP TRIGGER IF EXISTS update_devices_updated_at ON devices",
        """
        CREATE TRIGGER update_devices_updated_at
            BEFORE UPDATE ON devices
            FOR EACH ROW
            EXECUTE FUNCTION update_updated_at_column()
        """,
        "DROP TRIGGER IF EXISTS update_batches_updated_at ON batches",
        """
        CREATE TRIGGER update_batches_updated_at
            BEFORE UPDATE ON batches
            FOR EACH ROW
            EXECUTE FUNCTION update_updated_at_column()
        """
    ]
    
    try:
        for i, statement in enumerate(schema_statements):
            logger.info(f"Executing schema statement {i+1}/{len(schema_statements)}")
            execute_sql(rds_client, statement.strip())
        
        logger.info("Database schema created successfully")
        return True
    except Exception as e:
        logger.error(f"Failed to create database schema: {str(e)}")
        raise

def wait_for_cluster_ready(rds_client, max_retries=10, delay=30):
    """Wait for Aurora cluster to be ready"""
    import time
    
    for attempt in range(max_retries):
        try:
            logger.info(f"Checking cluster readiness (attempt {attempt + 1}/{max_retries})")
            
            # Try a simple query to test if cluster is ready
            execute_sql(rds_client, "SELECT 1 as test")
            logger.info("Cluster is ready!")
            return True
            
        except Exception as e:
            error_str = str(e)
            if "DatabaseNotFoundException" in error_str or "DBInstance" in error_str:
                logger.warning(f"Cluster not ready yet: {error_str}")
                if attempt < max_retries - 1:
                    logger.info(f"Waiting {delay} seconds before retry...")
                    time.sleep(delay)
                else:
                    logger.error("Cluster failed to become ready within timeout")
                    raise
            else:
                # Different error, don't retry
                logger.error(f"Unexpected error: {error_str}")
                raise
    
    return False

def lambda_handler(event, context):
    """
    Initialize database schema for IoT device bulk management using RDS Data API
    """
    logger.info("Starting database initialization with RDS Data API")
    logger.info(f"Event: {json.dumps(event)}")
    
    try:
        # Get RDS Data API client
        rds_client = get_rds_data_client()
        
        # Wait for cluster to be ready
        wait_for_cluster_ready(rds_client)
        
        # Create schema
        create_database_schema(rds_client)
        
        return {
            'statusCode': 200,
            'body': json.dumps({
                'message': 'Database schema initialized successfully using RDS Data API',
                'timestamp': datetime.utcnow().isoformat()
            })
        }
        
    except Exception as e:
        logger.error(f"Database initialization failed: {str(e)}")
        return {
            'statusCode': 500,
            'body': json.dumps({
                'error': str(e),
                'message': 'Database initialization failed'
            })
        }