#!/bin/bash

# AWS IoT Wireless Device Bulk Management Solution Deployment Script
# This script deploys the CDK solution with customer configuration

set -e

echo "🚀 AWS IoT Wireless Device Bulk Management Solution Deployment"
echo "=============================================================="

# Check prerequisites
echo "📋 Checking prerequisites..."

# Check if AWS CLI is installed and configured
if ! command -v aws &> /dev/null; then
    echo "❌ AWS CLI is not installed. Please install it first."
    echo "   Installation guide: https://docs.aws.amazon.com/cli/latest/userguide/getting-started-install.html"
    exit 1
fi

# Check if CDK is installed
if ! command -v cdk &> /dev/null; then
    echo "❌ AWS CDK is not installed. Installing now..."
    npm install -g aws-cdk
fi

# Check if Node.js is installed
if ! command -v node &> /dev/null; then
    echo "❌ Node.js is not installed. Please install Node.js 20+ first."
    exit 1
fi

# Check Node.js version
NODE_VERSION=$(node --version | cut -d'v' -f2 | cut -d'.' -f1)
if [ "$NODE_VERSION" -lt 20 ]; then
    echo "❌ Node.js version 20 or higher is required. Current version: $(node --version)"
    exit 1
fi

echo "✅ Prerequisites check passed"

# Check if config.json exists
if [ ! -f "config.json" ]; then
    echo "⚠️  config.json not found. Creating default configuration..."
    cat > config.json << EOF
{
  "notificationType": "SQS",
  "sqsProperties": {
    "queueName": "sidewalk-bulk-notifications",
    "visibilityTimeout": 300
  },
  "createWirelessDeviceApiTps": 10,
  "updateWirelessDeviceApiTps": 10,
  "getWirelessDeviceApiTps": 10
}
EOF
    echo "📝 Default config.json created. Please review and modify as needed."
    echo "   You can edit the TPS values based on your AWS IoT Core quotas."
fi



echo "📦 Installing dependencies..."
npm install

echo "🔨 Building TypeScript..."
npm run build

echo "🏗️  Bootstrapping CDK (if needed)..."
cdk bootstrap

echo "🚀 Deploying the solution..."
echo "   This will create the following resources:"
echo "   - S3 buckets for input and reports"
echo "   - Lambda functions for processing"
echo "   - Step Functions state machine"
echo "   - Aurora database cluster"
echo "   - CloudWatch dashboards"
echo "   - Notification services (SQS/SNS based on config)"

# Deploy with configuration
cdk deploy --require-approval never

echo ""
echo "✅ Deployment completed successfully!"
echo ""
echo "📊 Next Steps:"
echo "1. Note the S3 input bucket name from the CDK outputs above"
echo "2. Upload your device JSON files to the input bucket to start processing"
echo "3. Monitor progress in the CloudWatch dashboard"
echo "4. Check notifications in your configured SQS queue or SNS topic"
echo "5. Download detailed reports from the reports S3 bucket"
echo ""
echo "📖 Sample JSON files are provided:"
echo "   - sample-create-devices.json (for device provisioning)"
echo "   - sample-update-devices.json (for device configuration updates)"
echo ""
echo "🔧 To modify configuration:"
echo "   1. Edit config.json"
echo "   2. Run: cdk deploy"
echo ""
echo "🗑️  To remove the solution:"
echo "   Run: cdk destroy"
echo ""
echo "Happy device management! 🎉"